#!/usr/bin/env python3

from lib.repository import find_repository_root
import yaml
from copy import deepcopy
from pathlib import Path


class Config:
    def __init__(self, data):
        self._data = data

    def __getitem__(self, key):
        return self._data[key]

    def get_log_setting(self, setting_name, log_name, env, required=True):
        log_settings = self._find_log_settings(log_name)
        base_settings = self._data["log_settings"]
        merged_settings = _merge(log_settings, base_settings)
        setting_value = merged_settings.get(setting_name)
        if setting_value is None and required:
            raise RuntimeError('Setting "{}" not found for log "{}"'.format(setting_name, log_name))
        return self._expand_patterns(setting_value, log_name, env)

    def _find_log_settings(self, log_name):
        for log_settigns in self._data["logs"]:
            if log_settigns["name"] == log_name:
                return log_settigns
        return None

    def _expand_patterns(self, setting_value, log_name, env):
        if type(setting_value) == str:
            ret = setting_value.format(app=self._data["app"], env=env, log=log_name)
        elif type(setting_value) == dict:
            ret = {}
            for key, value in setting_value.items():
                ret[key] = self._expand_patterns(value, log_name, env)
        elif type(setting_value) == list:
            ret = []
            for value in setting_value:
                ret.append(self._expand_patterns(value, log_name, env))
        else:
            ret = setting_value
        return ret


def load_config(path):
    return Config(_load_config_as_dict(path))


def _load_config_as_dict(path):
    with open(path) as infile:
        data = yaml.safe_load(infile)
    if "base" in data:
        base_path = _resolve_config_path(Path(data["base"]))
        base = _load_config_as_dict(base_path)
        return _merge(data["config"], base)
    return data["config"]


def _resolve_config_path(path):
    if path.is_absolute():
        return path
    repo_path = find_repository_root(Path(".").resolve())
    path_from_repo_root = repo_path / path
    if path_from_repo_root.exists():
        return path_from_repo_root
    raise RuntimeError('Config "{}" not found'.format(path_from_repo_root))


def _merge(lhs, rhs):
    ret = {}
    for k, v in lhs.items():
        if type(v) == dict and k in rhs and type(rhs[k]) == dict:
            ret[k] = _merge(v, rhs[k])
        elif type(v) == list and k in rhs and type(rhs[k]) == list:
            ret[k] = deepcopy(v) + deepcopy(rhs[k])
        else:
            ret[k] = deepcopy(v)
    for k, v in rhs.items():
        if k not in lhs:
            ret[k] = deepcopy(v)
    return ret
