#!/usr/bin/env python3

from lib.aux import query_yes_no


class ConfigureOp:
    def __init__(self, logfeller, repo, config, auto_confirm):
        self._logfeller = logfeller
        self._repo = repo
        self._config = config
        self._auto_confirm = auto_confirm
        self._files_to_commit = set()

    def __call__(self):
        for env in self._config["envs"]:
            for log_config in self._config["logs"]:
                if self._need_configure(log_config, env) and (
                    self._auto_confirm or self._confirm(log_config, env)
                ):
                    if self._need_configure_parser(log_config, env):
                        self._configure_parser(log_config, env)
                    if self._need_configure_log(log_config, env):
                        self._configure_log(log_config, env)
                    if self._need_configure_stream(log_config, env):
                        self._configure_stream(log_config, env)
        if self._files_to_commit:
            self._repo.commit(self._files_to_commit, "Update logfeller configuration")

    def _need_configure(self, log_config, env):
        return (
            self._need_configure_parser(log_config, env)
            or self._need_configure_log(log_config, env)
            or self._need_configure_stream(log_config, env)
        )

    def _need_configure_parser(self, log_config, env):
        config_path = self._parser_config_path(log_config, env)
        if not self._logfeller.config_exists(config_path):
            return True
        fields = self._parser_fields(log_config, env)
        return sorted(fields) != sorted(log_config["fields"])

    def _parser_config_path(self, log_config, env):
        parser_name = self._parser_name(log_config, env)
        return self._logfeller.parser_config_path(parser_name)

    def _parser_name(self, log_config, env):
        return self._config.get_log_setting("logfeller_parser_name", log_config["name"], env)

    def _parser_fields(self, log_config, env):
        config_path = self._parser_config_path(log_config, env)
        return self._logfeller.get_fields_from_parser_config(config_path)

    def _need_configure_log(self, log_config, env):
        config_path = self._logs_config_path(log_config, env)
        if not self._logfeller.config_exists(config_path):
            raise RuntimeError('Logfeller logs config not found at "{}"'.format(config_path))
        log_name = self._config.get_log_setting("logfeller_log_name", log_config["name"], env)
        current_logs = self._logfeller.get_logs_from_logs_config(config_path)
        return log_name not in current_logs

    def _logs_config_path(self, log_config, env):
        config_name = self._config.get_log_setting(
            "logfeller_logs_config_name", log_config["name"], env
        )
        return self._logfeller.logs_config_path(config_name)

    def _need_configure_stream(self, log_config, env):
        config_path = self._streams_config_path(log_config, env)
        if not self._logfeller.config_exists(config_path):
            raise RuntimeError('Logfeller streams config not found at "{}"'.format(config_path))
        current_topics = self._logfeller.get_topics_from_streams_config(config_path)
        return self._topic(log_config, env) not in current_topics

    def _streams_config_path(self, log_config, env):
        config_name = self._config.get_log_setting(
            "logfeller_streams_config_name", log_config["name"], env
        )
        return self._logfeller.streams_config_path(config_name)

    def _topic(self, log_config, env):
        return self._config.get_log_setting("topic", log_config["name"], env)

    def _confirm(self, log_config, env):
        topic = self._topic(log_config, env)
        msg = 'Configure logfeller for topic "{}" in "{}" environment'.format(topic, env)
        return query_yes_no(msg)

    def _configure_parser(self, log_config, env):
        parser_name = self._parser_name(log_config, env)
        config_path = self._parser_config_path(log_config, env)
        if not self._logfeller.config_exists(config_path):
            self._logfeller.create_parser_config(log_config["fields"], config_path)
            time_format = self._time_format(log_config, env)
            self._logfeller.append_to_parsers_config(parser_name, time_format)
            self._logfeller.update_parsers_ya_make()
            self._files_to_commit.add(config_path)
            self._files_to_commit.add(self._logfeller.parsers_config_path())
            self._files_to_commit.add(self._logfeller.parsers_make_file_path())
        else:
            self._logfeller.rewrite_parser_config(log_config["fields"], config_path)
            self._files_to_commit.add(config_path)

    def _time_format(self, log_config, env):
        return self._config.get_log_setting("time_format", log_config["name"], env)

    def _configure_log(self, log_config, env):
        topic = self._topic(log_config, env)
        log_name = self._config.get_log_setting("logfeller_log_name", log_config["name"], env)
        config_path = self._logs_config_path(log_config, env)
        periods = self._config.get_log_setting("periods", log_config["name"], env, required=False)
        self._logfeller.append_to_logs_config(topic, log_name, config_path, periods)
        self._files_to_commit.add(config_path)

    def _configure_stream(self, log_config, env):
        topic = self._topic(log_config, env)
        parser_name = self._parser_name(log_config, env)
        config_path = self._streams_config_path(log_config, env)
        self._logfeller.append_to_streams_config(topic, parser_name, config_path)
        self._files_to_commit.add(config_path)


def configure_logfeller(logfeller, repo, config, auto_confirm):
    op = ConfigureOp(logfeller, repo, config, auto_confirm)
    op()
