#!/usr/bin/env python3

from lib.aux import query_yes_no


def configure_push_client(push_client, repo, config, app_path, auto_confirm):
    files_to_commit = set()
    for env in config["envs"]:
        for log_config in config["logs"]:
            config_path = _push_client_config_path(config, log_config["name"], app_path, env)
            topic = config.get_log_setting("topic", log_config["name"], env)
            options = config.get_log_setting(
                "push_client_log_options", log_config["name"], env, required=False
            )
            if not push_client.config_exists(config_path):
                if auto_confirm or _confirm_create_config(config_path, topic, env):
                    log_path = _log_path(config, log_config, env)
                    push_client.create_config(
                        _tvm_options(config, log_config, env),
                        topic,
                        log_path,
                        config_path,
                        options,
                    )
                    files_to_commit.add(config_path)
            elif topic not in push_client.get_topics_from_config(config_path):
                if auto_confirm or _confirm_add_to_config(config_path, topic, env):
                    log_path = _log_path(config, log_config, env)
                    push_client.append_topic_to_config(topic, log_path, config_path, options)
                    files_to_commit.add(config_path)
    if files_to_commit:
        repo.commit(files_to_commit, "Update push-client configuration")


def _push_client_config_path(config, log_name, app_path, env):
    local_path = config.get_log_setting("push_client_config_path", log_name, env)
    return "{}/{}".format(app_path, local_path).format(env=env)


def _confirm_create_config(config_path, topic, env):
    msg = 'Create config "{}" for topic "{}" in "{}" environment'.format(config_path, topic, env)
    return query_yes_no(msg)


def _log_path(config, log_config, env):
    return config.get_log_setting("fs_log_path", log_config["name"], env)


def _tvm_options(config, log_config, env):
    ret = {"id": config["tvm_id"][env]}
    tvm_secret_path = config.get_log_setting(
        "tvm_secret_path", log_config["name"], env, required=False
    )
    tvm_secret_value = config.get_log_setting(
        "tvm_secret_value", log_config["name"], env, required=False
    )
    if tvm_secret_path:
        ret["secret_path"] = tvm_secret_path
    elif tvm_secret_value:
        ret["secret_value"] = tvm_secret_value
    else:
        raise RuntimeError("TVM secret not specified")
    return ret


def _confirm_add_to_config(config_path, topic, env):
    msg = 'Add topic "{}" to config "{}" for "{}" environment'.format(topic, config_path, env)
    return query_yes_no(msg)
