#pragma once

#include <pgg/error.h>
#include <sharpei_client/errors.h>
#include <macs/errors.h>
#include <mail_errors/error_code.h>
#include <internal/common/error_enum.h>


namespace boost::system {

template <>
struct is_error_condition_enum<settings::Error> : std::true_type {};

} // boost::system

namespace settings {

using mail_errors::error_code;

class ErrorCategory final : public boost::system::error_category {
public:
    const char* name() const noexcept override {
        return "settings";
    }

    std::string message(int value) const override {
        switch (static_cast<Error>(value)) {
            case Error::ok:
                return "Ok";
            case Error::uidNotFound:
                return "User not found";
            case Error::sharpeiError:
                return "Failed sharpei request";
            case Error::deleteError:
                return "Failed to delete settings";
            case Error::updateError:
                return "Failed to update settings";
            case Error::nullValue:
                return "Fnvalid settings value in storage";
            case Error::getError:
                return "Failed to get settings";
            case Error::noSettings:
                return "Fettings not found";
            case Error::blackBoxError:
                return "Failed to perform request to blackbox";
            case Error::blackBoxUserError:
                return "User not found";
            case Error::invalidParameterError:
                return "Invalid parameter";
            case Error::blackBoxDefaultEmailError:
                return "Given default email was not validated";
            case Error::invalidFormatType:
                return "Invalid format type";
            case Error::noData:
                return "No data from database";
            case Error::noSuchNode:
                return "No such node";
        }
        return "Unknown error code: " + std::to_string(value);
    }

    bool equivalent(const boost::system::error_code& code, int condition) const noexcept override {
        switch (Error(condition)) {
            case Error::uidNotFound: {
                return code == sharpei::client::Errors::UidNotFound;
            }
            case Error::noSettings: {
                return code == pgg::error::CommonErrors::noDataReceived;
            }
            case Error::invalidParameterError: {
                return code == ::macs::error::Errors::invalidArgument;
            }
            default:
                return boost::system::error_category::equivalent(code, condition);
        }
    }
};

inline const ErrorCategory& getErrorCategory() {
    static ErrorCategory errorCategory;
    return errorCategory;
}

inline auto make_error_code(Error ec) {
    return error_code::base_type(static_cast<int>(ec), getErrorCategory());
}

inline auto make_error_condition(Error e) {
    return boost::system::error_condition(static_cast<int>(e), getErrorCategory());
}

} // namespace settings
