#pragma once

#include <yplatform/log.h>
#include <yplatform/task_context.h>
#include <yplatform/tskv/tskv.h>

#include <logdog/format/tskv.h>
#include <logdog/logger.h>
#include <logdog/attributes/mail_attributes.h>
#include <logdog/backend/yplatform_log.h>

#include <pgg/logging.h>
#include <pgg/logdog.h>

#include <internal/common/context.h>

namespace settings::logging {

constexpr static auto settings_formatter = logdog::tskv::make_formatter(BOOST_HANA_STRING("settings"));

LOGDOG_DEFINE_ATTRIBUTE(std::string, ctx_uid)

inline auto makeLogger() {
    return logdog::make_log(
        settings_formatter,
        yplatform::log::source {YGLOBAL_LOG_SERVICE, "settings"}
    );
}

inline auto makeLoggerWithRequestId(const std::string& requestId) {
    return logdog::bind(makeLogger(), logdog::request_id = requestId);
}

inline auto makeLoggerWithUid(const settings::ContextPtr& ctx) {
    return logdog::bind(makeLogger(), logdog::request_id = ctx->requestId(), logdog::uid = ctx->uid());
}

inline auto makeLoggerWithCtxUid(const settings::ContextPtr& ctx) {
    return logdog::bind(makeLogger(), logdog::request_id = ctx->uniq_id());
}

inline auto getPggLogger(const std::string& requestId) {
    return pgg::logging::makeTypedLog(
        logdog::bind(makeLoggerWithRequestId(requestId), logdog::message = "pgg")
    );
}

} // namespace settings::logging

#define SETTINGS_LOG_ERROR(ctx, ...) \
    LOGDOG_(::settings::logging::makeLoggerWithCtxUid(ctx), error, __VA_ARGS__)

#define SETTINGS_LOG_WARNING(ctx, ...) \
    LOGDOG_(::settings::logging::makeLoggerWithCtxUid(ctx), warning, __VA_ARGS__)

#define SETTINGS_LOG_DEBUG(ctx, ...) \
    LOGDOG_(::settings::logging::makeLoggerWithCtxUid(ctx), debug, __VA_ARGS__)

#define SETTINGS_CONTEXT_LOG_WARNING(ctx, ...) \
    LOGDOG_(::settings::logging::makeLoggerWithUid(ctx), warning, __VA_ARGS__)

#define SETTINGS_CONTEXT_LOG_DEBUG(ctx, ...) \
    LOGDOG_(::settings::logging::makeLoggerWithUid(ctx), debug, __VA_ARGS__)

#define SETTINGS_CONTEXT_LOG_ERROR(ctx, ...) \
    LOGDOG_(::settings::logging::makeLoggerWithUid(ctx), error, __VA_ARGS__)

#define SETTINGS_CONTEXT_LOG_NOTICE(ctx, ...) \
    LOGDOG_(::settings::logging::makeLoggerWithUid(ctx), notice, __VA_ARGS__)
