#pragma once

#include <macs/settings.h>

#include <map>
#include <memory>
#include <vector>
#include <optional>

namespace settings {

using Mode = ::macs::settings::Mode;
using SettingsList = ::macs::settings::SettingsList;
using SettingsListPtr = std::shared_ptr<SettingsList>;

struct Email {
    Email(): native(false), validated(false), def(false), rpop(false) {}
    Email (bool n, bool v, bool d, bool r, const std::string& addr, const std::string& dt)
            : native(n), validated(v), def(d), rpop(r), address(addr), date(dt) {}
    bool native;
    bool validated;
    bool def;
    bool rpop;
    std::string address;
    std::string date;
};

using MacsSignature = ::macs::settings::Signature;
using MacsSignaturesList = ::macs::settings::SignaturesList;
using MacsSignaturesListOpt = ::macs::settings::SignaturesListOpt;

struct Signature : MacsSignature {
    Signature() = default;
    Signature(const MacsSignature& s) : MacsSignature(s) {}
    Signature(MacsSignature&& s) : MacsSignature(std::move(s)) {}

    std::optional<bool> is_sanitize = false;
};

using SignaturesList = std::vector<Signature>;
using SignaturesListOpt = std::optional<SignaturesList>;

using SettingsMap = ::macs::settings::SettingsMap;
using Profile = ::macs::settings::Profile;
using Parameters = ::macs::settings::Parameters;
using Settings = ::macs::settings::Settings;

struct MapOptions {
    MapOptions(const MapOptions&) = delete;
    MapOptions(MapOptions&&) = default;

    MapOptions() {}
    MapOptions(Parameters&& parameters)
        : single_settings(std::move(parameters.single_settings)) {}
    MapOptions(Profile&& profile) : single_settings(std::move(profile.single_settings)) {
        if (profile.signs) {
            signs = SignaturesList {};
            signs->reserve(profile.signs->size());
            std::move(profile.signs->begin(), profile.signs->end(), std::back_inserter(*signs));
        }
    }
    MapOptions(const SignaturesListOpt& signs_) {
        if(signs_) {
            signs = SignaturesList {};
            signs->reserve(signs_->size());
            std::copy(signs_->begin(), signs_->end(), std::back_inserter(*signs));
        }
    }

    bool empty() const {
        return single_settings.empty() && !emails.has_value() && !signs.has_value();
    }

    SettingsMap single_settings;
    std::optional<std::vector<Email>> emails;
    SignaturesListOpt signs;
    std::optional<bool> error;
};

using MapOptionsPtr = std::shared_ptr<MapOptions>;

struct DoubleMapOptions {

    DoubleMapOptions(Settings&& settings)
        : profile(std::move(settings.profile))
        , parameters(std::move(settings.parameters)) {}

    DoubleMapOptions() = default;
    DoubleMapOptions(const DoubleMapOptions&) = delete;
    DoubleMapOptions(DoubleMapOptions&&) = default;

    MapOptions profile;
    MapOptions parameters;
};

using TextTraits = ::macs::settings::TextTraits;

using ProfilePtr = ::macs::settings::ProfilePtr;
using ParametersPtr = ::macs::settings::ParametersPtr;
using SettingsPtr = ::macs::settings::SettingsPtr;

enum class DatabaseRole {
    Master,
    Replica,
};

} // namespace settings
