#pragma once

#include <internal/common/types_reflection.h>

#include <internal/http/detail/format.h>
#include <internal/http/detail/handlers/base.h>
#include <internal/http/detail/handlers/utils.h>

namespace settings::http::detail::handlers {

template <class T, template <class> class TransferEncodedT>
class BaseGet : public Base {
public:
    using Settings = T;
    template <class F> using TransferEncoded = TransferEncodedT<F>;
    using Result = ResultSettings<Settings>;

    BaseGet(const std::string& logicModule) : Base(logicModule) {}

    virtual ymod_webserver::methods::http_method method() const override {
        return ymod_webserver::methods::mth_get;
    }

protected:
    virtual expected<Settings> get(RequestPtr request, ContextPtr ctx) const = 0;

private:
    expected<void> invoke(RequestPtr request, ResponsePtr response, ContextPtr ctx) const override;

    template <class Formatted>
    static void write(const ResponsePtr& response, Formatted formatted);
};

template <class T, template <class> class E>
expected<void> BaseGet<T, E>::invoke(RequestPtr request, ResponsePtr response, ContextPtr ctx) const {
    return getFormat(request->url.params)
        .bind([&] (auto&& format) {
            return get(request, ctx)
                .bind([&](auto&& settings) -> expected<void> {
                    switch (format) {
                        case format::Type::Json:
                            response->set_code(ymod_webserver::codes::ok);
                            response->set_content_type("application/json");
                            write(response, format::json(Result {std::move(settings)}));
                            break;
                        default:
                            return make_unexpected(
                                error_code(make_error_code(Error::invalidFormatType))
                            );
                    }
                    return {};
                });
        });
}

template <class T, template <class> class E>
template <class Formatted>
void BaseGet<T, E>::write(const ResponsePtr& response, Formatted formatted) {
    TransferEncoded<Formatted>(std::move(formatted))(*response);
}

} // namespace settings::http::detail::handlers
