#pragma once

#include <map>
#include <string>

#include <butil/StrUtils/Iconv.h>

#include "utf.h"

/**
 * Text converter interface.
 */
struct TextConverter {
    /**
     * Converts "in" text from specified "charset" to "out" UTF-8.
     */
    virtual void convert( const std::string & in, std::string & out,
            const std::string & charset ) const = 0;
    virtual ~TextConverter() {}
protected:
    void convert_core( const std::string & in, std::string & out,
            const std::string & from, const std::string to = "utf-8" ) const
    {
        out.clear();
        out.reserve( in.size() );
        if (Iconv::recode(from.c_str(), to.c_str(), in, out, true) != ICONV_OK) {
            throw std::runtime_error("Convertion failed");
        }
    }
};

/**
 * Normal text convertor to UTF-8 via iconv.
 */
struct NormalTextConverter : public TextConverter {
    virtual void convert( const std::string & in, std::string & out,
            const std::string & charset ) const {
        convert_core( in, out, charset );
    }
};

/**
 * Special implementation for the UTF-8 to UTF-8 conversion case.
 */
struct UtfTextConverter : public TextConverter {
    virtual void convert( const std::string & in, std::string & out,
            const std::string & /*charset*/ ) const {
        out = get_validated_utf8( in );
    }
};

/**
 * Parent for the double coded text converters' implementations
 */
struct DoubleTextConverter : public TextConverter {
    void recodeDouble( const std::string & in, std::string & out,
            const std::string & codepage, const std::string & middleCodepage ) const {
        convert_core( in, out, codepage, middleCodepage );
        std::string buffer;
        std::swap(buffer,out);
        convert_core( buffer, out, codepage );
    }
};

/**
 * Implementation for the double coded koi-8 text
 */
struct KoiKoiTextConverter : public DoubleTextConverter {
    virtual void convert( const std::string & in, std::string & out,
            const std::string & ) const {
        return recodeDouble(in, out, "koi8-r", "cp1251");
    }
};

/**
 * Implementation for the double coded cp1251 text
 */
struct WinWinTextConverter : public DoubleTextConverter {
    virtual void convert( const std::string & in, std::string & out,
            const std::string & ) const {
        return recodeDouble(in, out, "cp1251", "koi8-r");
    }
};
