#pragma once

#include <stdexcept>

template<class iterator>
bool is_valid_utf8_char(iterator it, iterator end) noexcept
{
    bool res = true;
    iterator tmp = it;
    unsigned char ch = static_cast<unsigned char>(*it);
    if ((ch & 0x80) == 0)  // U+0000-U+007F (ASCII)
        return res;

    if (ch == 0xC0 || ch == 0xC1) // Overlong encoding
        return false;

    wchar_t guard_10FFFF = (static_cast<unsigned char>(*tmp) & 0x07) << 2;
    res &= ((static_cast<unsigned char>(*++tmp) & 0xC0) == 0x80 && tmp != end);
    if ((ch & 0xE0) == 0xC0) // U+0080-U+07FF
        return res;

    guard_10FFFF += (static_cast<unsigned char>(*tmp) & 0x30) >> 4;
    res &= (((static_cast<unsigned char>(*++tmp)) & 0xC0) == 0x80 && tmp != end);
    if ((static_cast<unsigned char>(*it) & 0xF0) == 0xE0) // U+0800-U+FFFF
        return res;

    res &= ((static_cast<unsigned char>(*++tmp) & 0xC0) == 0x80 && tmp != end);
    if ((ch & 0xF8) == 0xF0) { // U+10000-U+10FFFF
        return res && (guard_10FFFF <= 0x10); // ensuring a unicode codepoint does not exceed 10FFFF
    }

    return false;
}

/*
    The following functions assume the utf8 char sequence is complete
    (i.e. if there's a beginning of the utf8-encoding of a unicode code point in a sequence, the sequence must contain the rest of this encoding);

*/
template<class iterator>
iterator next_utf8_char(iterator it)
{
    iterator orit = it;
    ++it;
    if ((static_cast<unsigned char>(*orit) & 0x80) == 0) // U+0000-U+007F (ansi)
        return it;
    ++it;
    if ((static_cast<unsigned char>(*orit) & 0xE0) == 0xC0) // U+0080-U+07FF
        return it;
    ++it;
    if ((static_cast<unsigned char>(*orit) & 0xF0) == 0xE0) // U+0800-U+FFFF
        return it;
    ++it;
    if ((static_cast<unsigned char>(*orit) & 0xF8) == 0xF0) // U+10000-U+10FFFF
        return it;
    throw std::runtime_error("bad utf: next_utf8_char failed");
}

template <typename In, typename Out>
void validateUtf8 ( In first, In last, Out out )
{
    while (first != last) {
        if( is_valid_utf8_char(first,last) ) {
            In next = next_utf8_char(first);
            out = std::copy(first, next, out);
            first = next;
        } else {
            ++first;
            *out++='?';
        }
    }
}

inline
void validate_utf8 (std::string &src)
{
    validateUtf8 ( src.begin(), src.end(), src.begin() );
}

inline
std::string get_validated_utf8( const std::string & src )
{
    std::string retval;
    retval.reserve(src.capacity());
    validateUtf8 ( src.begin(), src.end(), std::back_inserter(retval) );
    return retval;
}
