#include <yamail/data/deserialization/ptree.h>

#include <yplatform/find.h>
#include <yplatform/module_registration.h>

#include <internal/api/config.h>
#include <internal/api/config_reflection.h>
#include <internal/api/impl.h>
#include <internal/api/utils.h>

#include <internal/common/error_code.h>
#include <internal/common/logger.h>

#include <internal/recognizer/UTFizer.h>

#include <boost/range/algorithm.hpp>
#include <boost/range/adaptor/filtered.hpp>

namespace settings::api {

void api_impl::init(const yplatform::ptree& ptree) {
    using yamail::data::deserialization::fromPtree;
    Config config = fromPtree<Config>(ptree);
    mode = config.mode;
}

void api_impl::utfizeSigns(MapOptions& options) const {
    if (options.signs) {
        boost::for_each(*options.signs
            | boost::adaptors::filtered([](auto& v) { return v.text.has_value() && !v.text->empty();}),
            [&](auto& s) {
                s.text = utfizer->utfize(*s.text);
                s.text_traits = std::make_from_tuple<TextTraits>(utfizer->recognize(*s.text));
            }
        );
    }
}

expected<MapOptions> api_impl::getProfile(ContextPtr ctx) {
    SETTINGS_CONTEXT_LOG_NOTICE(ctx, logdog::where_name = "/get_profile");
    if (ctx->askValidator()) {
        return blackbox->getAccountInfo(ctx, ctx->uid(), ctx->userIp())
            .bind([&](auto&& accauntInfo) {
                return asyncGetProfile(ctx, accauntInfo, ctx->settingsList());
            });
    } else {
        return asyncGetProfile(ctx, blackbox::AccountInfoPtr {}, ctx->settingsList());
    }
}

expected<MapOptions> api_impl::asyncGetProfile(ContextPtr ctx, AccountInfoPtr accountInfo,
        const SettingsList& settings) const {
    return macs->getProfile(ctx, settings)
        .bind([&] (auto&& profile) {
            auto options = MapOptions {std::move(profile)};
            addEmails(options, accountInfo, settings);
            addFromName(options, accountInfo, settings, mode);
            addDefaultEmail(options, accountInfo, settings);
            utfizeSigns(options);
            return options;
        });
}

expected<MapOptions> api_impl::getParams(ContextPtr ctx) {
    SETTINGS_CONTEXT_LOG_NOTICE(ctx, logdog::where_name = "/get_params");
    return macs->getParameters(ctx, ctx->settingsList())
        .bind([&] (auto&& parameters) {
            return MapOptions {std::move(parameters)};
        });
}

expected<void> api_impl::updateProfile(ContextPtr ctx) {
    SETTINGS_CONTEXT_LOG_NOTICE(ctx, logdog::where_name = "/update_profile");
    auto options = ctx->settings();

    auto defaultEmailIt = options->single_settings.find("default_email");

    if (defaultEmailIt != options->single_settings.end()) {
        auto  accountInfoResult = blackbox->getAccountInfo(ctx, ctx->uid(), ctx->userIp());
        if (!accountInfoResult) {
            return make_unexpected(accountInfoResult.error());
        }
        if (!accountInfoResult.value()->validateDefaultEmail(defaultEmailIt->second)) {
            return make_unexpected<mail_errors::error_code>(make_error_code(Error::blackBoxDefaultEmailError));
        }
    }

    return asyncUpdateProfile(ctx, options);
}

expected<void> api_impl::asyncUpdateProfile(ContextPtr ctx, MapOptionsPtr options) const {
    MacsSignaturesListOpt macsSigns;
    getMacsSignatures(options->signs, macsSigns);

    auto profile = std::make_shared<Profile>(
        std::move(macsSigns),
        std::move(options->single_settings)
    );
    return macs->updateProfile(ctx, profile)
        .bind([&] (auto&& result) -> expected<void> {
            if (!result) {
                return initSettings(ctx)
                    .bind([&](auto&&) -> expected<void> {
                        return macs->updateProfile(ctx, profile)
                            .bind([&](auto&& result) -> expected<void> {
                                if (!result) {
                                    return make_unexpected<mail_errors::error_code>(make_error_code(Error::updateError));
                                }
                                userJournal->asyncLogSettingsUpdate(ctx, std::move(*profile));
                                return {};
                            });
                    });
            }
            userJournal->asyncLogSettingsUpdate(ctx, std::move(*profile));
            return {};
        });
}

expected<void> api_impl::updateParams(ContextPtr ctx) {
    SETTINGS_CONTEXT_LOG_NOTICE(ctx, logdog::where_name = "/update_params");
    auto options = ctx->settings();
    auto parameters = std::make_shared<Parameters>(options->single_settings);
    return macs->updateParameters(ctx, parameters)
        .bind([&] (auto&& result) -> expected<void> {
            if (!result) {
                return initSettings(ctx)
                    .bind([&](auto&&) -> expected<void> {
                        return macs->updateParameters(ctx, parameters)
                            .bind([&](auto&& result) -> expected<void> {
                                if (!result) {
                                    return make_unexpected<mail_errors::error_code>(make_error_code(Error::updateError));
                                }
                                userJournal->asyncLogSettingsUpdate(ctx, std::move(*parameters));
                                return {};
                            });
                    });
            }
            userJournal->asyncLogSettingsUpdate(ctx, std::move(*parameters));
            return {};
        });
}

expected<void> api_impl::updateProtectedParams(ContextPtr ctx) {
    SETTINGS_CONTEXT_LOG_NOTICE(ctx, logdog::where_name = "/update_protected_params");
    auto options = ctx->settings();
    auto parameters = std::make_shared<Parameters>(options->single_settings);
    return macs->updateProtectedParameters(ctx, parameters)
        .bind([&] (auto&& result) -> expected<void> {
            if (!result) {
                return initSettings(ctx)
                    .bind([&](auto&&) -> expected<void> {
                        return macs->updateProtectedParameters(ctx, parameters)
                            .bind([&](auto&& result) -> expected<void> {
                                if (!result) {
                                    return make_unexpected<mail_errors::error_code>(make_error_code(Error::updateError));
                                }
                                userJournal->asyncLogSettingsUpdate(ctx, std::move(*parameters));
                                return {};
                            });
                    });
            }
            userJournal->asyncLogSettingsUpdate(ctx, std::move(*parameters));
            return {};
        });
}

expected<DoubleMapOptions> api_impl::get(ContextPtr ctx) {
    SETTINGS_CONTEXT_LOG_NOTICE(ctx, logdog::where_name = "/get");
    if (ctx->askValidator()) {
        return blackbox->getAccountInfo(ctx, ctx->uid(), ctx->userIp())
            .bind([&](auto&& accauntInfo) {
                return asyncGet(ctx, accauntInfo, ctx->settingsList());
            });
    } else {
        return asyncGet(ctx, blackbox::AccountInfoPtr {}, ctx->settingsList());
    }
}

expected<DoubleMapOptions> api_impl::asyncGet(ContextPtr ctx, AccountInfoPtr accountInfo,
        const SettingsList& settingsList) const {
    return macs->getSettings(ctx, settingsList)
        .bind([&](auto&& settings) {
            auto options = DoubleMapOptions {std::move(settings)};
            addDefaultEmail(options.profile, accountInfo, settingsList);
            addEmails(options.profile, accountInfo, settingsList);
            addFromName(options.profile, accountInfo, settingsList, mode);
            utfizeSigns(options.profile);
            return options;
        });
}

expected<bool> api_impl::initSettings(ContextPtr ctx) const {
    return blackbox->isUserExists(ctx, ctx->uid(), ctx->userIp())
        .bind([&]() {
            return macs->initSettings(ctx);
        });
}

expected<void> api_impl::remove(ContextPtr ctx) {
    SETTINGS_CONTEXT_LOG_NOTICE(ctx, logdog::where_name = "/remove");
    return macs->deleteSettings(ctx)
        .bind([](auto&& result) -> expected<void> {
            if (!result) {
                return make_unexpected<mail_errors::error_code>(make_error_code(Error::deleteError));
            }
            return {};
        });
}

expected<void> api_impl::deleteParams(ContextPtr ctx) {
    SETTINGS_CONTEXT_LOG_NOTICE(ctx, logdog::where_name = "/delete_params");
    return macs->eraseParameters(ctx, ctx->settingsList())
        .bind([&](auto&& result) -> expected<void> {
            if (!result) {
                return make_unexpected<mail_errors::error_code>(make_error_code(Error::deleteError));
            }
            return {};
        });
}

api_impl::api_impl()
        : macs(yplatform::find<macs::Interface, std::shared_ptr>("macs")),
          blackbox(yplatform::find<blackbox::Interface, std::shared_ptr>("settings_blackbox")),
          userJournal(yplatform::find<user_journal::Interface, std::shared_ptr>("settings_user_journal")),
          utfizer(std::make_shared<utfizer::UTFizer>()) {
}

api_impl::api_impl(MacsPtr macs_, BlackBoxPtr blackbox_,
        UserJournalPtr userJournal_, UTFizerPtr utfizer_)
        : macs(macs_),
          blackbox(blackbox_),
          userJournal(userJournal_),
          utfizer(utfizer_) {
}

}

DEFINE_SERVICE_OBJECT(settings::api::api_impl)
