#include <boost/algorithm/string/predicate.hpp>

#include <internal/blackbox/account_info.h>

namespace settings::blackbox {
namespace detail {

void splitDomain(const std::string &email, std::string &login, std::string &domain) {
    auto at = email.find('@');
    if (at == std::string::npos) {
        login = email;
        domain.clear();
        return;
    }
    login = email.substr(0, at);
    domain = email.substr(at + 1);
}

std::string loginToCanonicalForm(const std::string& login) {
    auto res = login.substr(0, login.find('+'));
    std::replace(res.begin(), res.end(), '-', '.');
    return res;
}

bool loginsEquals(const std::string& login1, const std::string& login2) {
    return boost::algorithm::iequals(loginToCanonicalForm(login1), loginToCanonicalForm(login2));
}

class IsValidatedPred {
public:
    IsValidatedPred(const std::string& email) {
        splitDomain( email, login_, domain_ );
    }

    bool operator ()(const Email& it) {
        std::string login, domain;
        splitDomain(it.address, login, domain);
        return it.validated
                && boost::algorithm::iequals(login, login_)
                && boost::algorithm::iequals(domain, domain_);
    }

private:
    std::string login_;
    std::string domain_;
};

class YandexIsValidatedPred {
public:
    YandexIsValidatedPred(const std::string& email) {
        splitDomain(email, login_, domain_);
    }

    bool operator()(const Email& it) {
        std::string login, domain;
        splitDomain(it.address, login, domain);
        return it.native
                && loginsEquals(login, login_)
                && boost::algorithm::iequals(domain, domain_);
    }

private:
    std::string login_;
    std::string domain_;
};

} // namespace detail

AccountInfo::AccountInfo(const std::string& firstName, const std::string& lastName,
                         const std::string& firstNameEng, const std::string& lastNameEng, const AddressList& emailList)
        : fromName_(makeFromName(firstName, lastName)), fromNameEng_(makeFromName(firstNameEng, lastNameEng)) {
    const auto defaultAddress = std::find_if(emailList.begin(), emailList.end(),
                                           std::mem_fn(&Address::isDefault));
    if (defaultAddress == emailList.end()) {
        throw std::runtime_error("No default email found");
    }

    for (const auto& item : emailList) {
        emails_.items.emplace_back(Email(item.native, item.validated, item.def,
                                                   item.rpop, item.address, item.date));
    }
    emails_.defaultEmail = emails_.items.begin() + (defaultAddress - emailList.begin());
}

const EmailList& AccountInfo::emails() const {
    return emails_;
}

const std::string& AccountInfo::fromName() const {
    return fromName_;
}

const std::string& AccountInfo::fromNameEng() const {
    return fromNameEng_;
}

bool AccountInfo::validateDefaultEmail(const std::string& email) const {
    const auto validated = std::find_if(emails_.items.begin(), emails_.items.end(),
                                        detail::YandexIsValidatedPred(email));
    if (validated == emails_.items.end()) {
        return validateReplyTo(email);
    }
    return true;
}

bool AccountInfo::validateReplyTo(const std::string& email) const {
    return std::find_if(emails_.items.begin(), emails_.items.end(),
                        detail::IsValidatedPred(email)) != emails().items.end();
}

std::string AccountInfo::makeFromName(const std::string& firstName, const std::string& lastName) {
    return firstName.empty() || lastName.empty()
            ? firstName + lastName
            : firstName + ' ' + lastName;
}

} // namespace settings::blackbox
