#include <yplatform/find.h>
#include <tvm_guard/types.h>
#include <butil/http/headers.h>

#include <internal/blackbox/http.h>
#include <internal/common/error.h>
#include <internal/common/error_code.h>

namespace {

inline std::string makeHeaders(const std::string& ticket) {
    http::headers headers;
    headers.add(tvm_guard::header::serviceTicket(), ticket);
    return headers.format();
}

}

namespace settings::blackbox {

Http::Http(HttpClientPtr httpClient, Tvm2ModulePtr tvm2Module)
        : httpClient_(std::move(httpClient)),
          tvm2Module_(std::move(tvm2Module)) { }

expected<std::string> Http::infoRequest(ContextPtr ctx, const std::string& url) const {
    std::string ticket;
    auto error = tvm2Module_->get_service_ticket("blackbox", ticket);
    if (error) {
        return make_unexpected(error_code(
            make_error_code(Error::blackBoxError), error.message())
        );
    }

    auto request = yhttp::request::GET(url, makeHeaders(ticket));

    boost::system::error_code ec;

    auto response = httpClient_->async_run(ctx, std::move(request), ctx->yield()[ec]);

    if (ec) {
        return make_unexpected(error_code(ec));
    }
    if (response.status != 200) {
        return make_unexpected(
            error_code(
                make_error_code(Error::blackBoxError),
                "request to blackbox ended with not 200 http status code"
            )
        );
    }
    return response.body;
}

} // namespace settings::blackbox
