#include <internal/blackbox/types_reflection.h>
#include <internal/blackbox/impl.h>
#include <internal/blackbox/http.h>
#include <internal/common/logger.h>
#include <internal/common/error_code.h>

#include <yamail/data/deserialization/json_reader.h>

namespace settings::blackbox {

using namespace std::string_literals;

Impl::Impl(HttpPtr http) : http_(http) { }

void Impl::init() {
    http_ = std::make_shared<Http>(
        yplatform::find<ymod_httpclient::cluster_call, std::shared_ptr>("blackbox_client"),
        yplatform::find<ymod_tvm::tvm2_module, std::shared_ptr>("ymod_tvm")
    );
}

expected<void> Impl::isUserExists(ContextPtr ctx, const std::string& uid, const std::string& userIp) const {
    std::string requestUrl = ymod_httpclient::url_encode({
        {"method",      "userinfo"},
        {"format",      "json"},
        {"uid",         uid},
        {"userip",      userIp},
        {"dbfields",    suidKey.c_str()}
    }, '?');
    return http_->infoRequest(ctx, requestUrl)
        .bind([&] (auto&& response) -> expected<void> {
            InfoResponse infoResponse;
            try {
                yamail::data::deserialization::fromJson<InfoResponse>(response, infoResponse);
                if (infoResponse.users.empty()) {
                    return make_unexpected(error_code(
                        make_error_code(Error::blackBoxUserError), "error parse blackbox result")
                    );
                }
                User user = infoResponse.users.front();
                if (!user.uid.has_value() || user.uid->value.empty()) {
                    return make_unexpected(error_code(
                        make_error_code(Error::blackBoxUserError), "empty uid in result")
                    );
                }
                if (!user.dbfields.has_value() || user.dbfields.value()[suidKey.c_str()].empty()) {
                    return make_unexpected(error_code(
                        make_error_code(Error::blackBoxUserError), "empty suid in result")
                    );
                }
            } catch (const std::exception &exc) {
                SETTINGS_CONTEXT_LOG_ERROR(
                    ctx,
                    logdog::exception = exc
                );
                return make_unexpected(error_code(
                    make_error_code(Error::blackBoxUserError), "error parse blackbox result")
                );
            }
            return {};
        });
}

expected<AccountInfoPtr> Impl::getAccountInfo(ContextPtr ctx, const std::string& uid, const std::string& userIp) const {
    std::string requestUrl = ymod_httpclient::url_encode({
        {"method",      "userinfo"},
        {"format",      "json"},
        {"emails",      "getall"},
        {"uid",         uid},
        {"userip",      userIp},
        {"dbfields",    firstNameKey.c_str() + ","s + lastNameKey.c_str()},
        {"attributes",  firstNameEngKey.c_str() + ","s + lastNameEngKey.c_str()}
    }, '?');
    return http_->infoRequest(ctx, requestUrl)
        .bind([&] (auto&& response) -> expected<AccountInfoPtr> {
            InfoResponse infoResponse;
            try {
                yamail::data::deserialization::fromJson<InfoResponse>(response, infoResponse);
                if (infoResponse.users.empty()) {
                    return make_unexpected(error_code(
                        make_error_code(Error::blackBoxUserError), "error parse blackbox result")
                    );
                }
                User user = infoResponse.users.front();
                if (!user.uid.has_value() || user.uid->value.empty()) {
                    return make_unexpected(error_code(
                        make_error_code(Error::blackBoxUserError), "empty uid in result")
                    );
                }
                if (!user.dbfields.has_value()) {
                    return make_unexpected(error_code(
                        make_error_code(Error::blackBoxUserError), "error parse blackbox result")
                    );
                }
                std::string firstNameEngKeyValue;
                std::string lastNameEngKeyValue;
                if (user.attributes.has_value()) {
                    firstNameEngKeyValue = user.attributes.value()[firstNameEngKey.c_str()];
                    lastNameEngKeyValue = user.attributes.value()[lastNameEngKey.c_str()];
                }
                return std::make_shared<AccountInfo>(
                    user.dbfields.value()[firstNameKey.c_str()],
                    user.dbfields.value()[lastNameKey.c_str()],
                    firstNameEngKeyValue,
                    lastNameEngKeyValue,
                    user.addresses.value()
                );
            } catch (const std::exception &exc) {
                SETTINGS_CONTEXT_LOG_ERROR(
                    ctx,
                    logdog::exception = exc
                );
                return make_unexpected(error_code(
                    make_error_code(Error::blackBoxUserError), "error parse blackbox result")
                );
            }
        });
}

using impl = Impl;

} // namespace settings::blackbox

#include <yplatform/module_registration.h>

DEFINE_SERVICE_OBJECT(settings::blackbox::impl)
