#include <internal/common/error.h>
#include <internal/http/detail/handlers/base.h>
#include <internal/http/detail/handlers/utils.h>

namespace settings::http::detail::handlers {

namespace {

inline std::string errorText(const std::string& message, const std::string& uniqId) {
    return  message + "\nrequest id: " + uniqId;
}

inline void setResponse(
        ResponsePtr response,
        ymod_webserver::codes::code code,
        const std::string& contentType,
        const std::string& resultBody
) {
    response->set_code(code);
    response->set_content_type(contentType);
    response->result_body(resultBody);
}

}

const LogicAdaptor& Base::logic() const {
    return logic_;
}

void Base::process(RequestPtr request, ResponsePtr response, tvm2::TvmGuardPtr tvmGuardPtr, ContextPtr ctx) const {
    if (request->method != method()) {
        response->result(ymod_webserver::codes::method_not_allowed);
        return;
    }

    try {
        const auto tvm = tvmGuardPtr->check(
            uri(),
            getUid(request),
            getOptionalHeader(request, tvm_guard::header::serviceTicket()),
            getOptionalHeader(request, tvm_guard::header::userTicket())
        );
        if (tvm.action == tvm_guard::Action::reject) {
            SETTINGS_LOG_ERROR(
            ctx,
            logdog::message = "tvm_guard rejected by reason: " + tvm_guard::toString(tvm.reason));
            response->result(ymod_webserver::codes::unauthorized);
            return;
        }

        auto result = invoke(request, response, ctx);
        if (!result) {
            using settings::Error;
            auto ec = std::move(result).error();
            SETTINGS_CONTEXT_LOG_ERROR(
                ctx,
                logdog::error_code = ec
            );
            if (ec == Error::uidNotFound ||
                    ec == Error::blackBoxUserError) {
                setResponse(
                    response,
                    ymod_webserver::codes::not_found,
                    "text/plain",
                    errorText("User not found", ctx->uniq_id())
                );
            } else if (ec == Error::invalidParameterError) {
                setResponse(
                    response,
                    ymod_webserver::codes::bad_request,
                    "text/plain",
                    errorText(ec.message(), ctx->uniq_id())
                );
            } else {
                setResponse(
                    response,
                    getYmodWebserverCode(ec),
                    "text/plain",
                    errorText(ec.message(), ctx->uniq_id())
                );
            }
        }
    } catch (const boost::coroutines::detail::forced_unwind&) {
        throw;
    } catch (const std::exception& error) {
        logError(ctx, error);
        setResponse(
            response,
            ymod_webserver::codes::internal_server_error,
            "text/plain",
            errorText("Internal error", ctx->uniq_id())
        );
    } catch (...) {
        logUnknownError(ctx);
        setResponse(
            response,
            ymod_webserver::codes::internal_server_error,
            "text/plain",
            errorText("Internal error", ctx->uniq_id())
        );
    }
}

} //namespace settings::http::detail::handlers
