#include <yamail/data/deserialization/json_reader.h>

#include <internal/common/types_reflection.h>
#include <internal/http/detail/handlers/utils.h>

#include <boost/algorithm/string.hpp>
#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/algorithm.hpp>
#include <boost/regex.hpp>

namespace settings::http::detail::handlers {

ContextId getContextId(const Request& request) {
    return request.context->uniq_id();
}

RequestId getRequestId(const Request& request) {
    return getHeader(request, "x-request-id", "");
}

UserIp getUserIp(const Request& request) {
    return getHeader(request, "x-real-ip", request.context->remote_address);
}

ClientType getClientType(const Request& request) {
    return getHeader(request, "x-yandex-clienttype", "");
}

ClientVersion getClientVersion(const Request& request) {
    return getHeader(request, "x-yandex-clientversion", "");
}

TestBuckets getTestBuckets(const Request& request) {
    return getHeader(request, "x-yandex-expboxes", "");
}

EnabledTestBuckets getEnabledTestBuckets(const Request& request) {
    return getHeader(request, "x-yandex-enabledexpboxes", "");
}

ContextPtr makeContext(const RequestPtr& request, YieldCtx yc) {
    return boost::make_shared<context>(
        getContextId(*request),
        getRequestId(*request),
        getUserIp(*request),
        getClientType(*request),
        getClientVersion(*request),
        getTestBuckets(*request),
        getEnabledTestBuckets(*request),
        yc
    );
}

std::string getHeader(const Request& request, const std::string& headerName, const std::string& defaultValue) {
    const auto header = request.headers.find(boost::algorithm::to_lower_copy(headerName));
    if (header != request.headers.end()) {
        return header->second;
    }
    return defaultValue;
}

std::optional<std::string> getOptionalHeader(const RequestPtr& request, const std::string& headerName) {
    const auto header = request->headers.find(boost::algorithm::to_lower_copy(headerName));
    return header != request->headers.end() ?  std::make_optional(header->second) : std::nullopt;
}

Uid getUid(const RequestPtr& request) {
    auto it = request->url.params.find("uid");
    return it == request->url.params.end() ? "" : it->second;
}

ymod_webserver::codes::code getYmodWebserverCode(const settings::Error& value) {
    using settings::Error;
    switch (value) {
        case Error::ok:
            return ymod_webserver::codes::ok;
        case Error::uidNotFound:
        case Error::blackBoxUserError:
        case Error::noSettings:
        case Error::noData:
            return ymod_webserver::codes::not_found;
        case Error::blackBoxDefaultEmailError:
        case Error::invalidParameterError:
        case Error::invalidFormatType:
        case Error:: noSuchNode:
            return ymod_webserver::codes::bad_request;
        case Error::getError:
        case Error::updateError:
        case Error::deleteError:
        case Error::nullValue:
        case Error::blackBoxError:
        case Error::sharpeiError:
            return ymod_webserver::codes::internal_server_error;
    }
    return ymod_webserver::codes::internal_server_error;
}

ymod_webserver::codes::code getYmodWebserverCode(const mail_errors::error_code& value) {
    if (value.category() == settings::getErrorCategory()) {
        return getYmodWebserverCode(settings::Error(value.value()));
    }
    return ymod_webserver::codes::internal_server_error;
}

bool isValidParamName(const std::string& name) {
    static const boost::regex regex(R"(^[A-Za-z][^=]*$)");
    return boost::regex_match(name, regex);
}

bool isValidSettings(const std::string& name) {
    return name != "uid"
        && name != "suid"
        && name != "mdb"
        && name != "ask_validator"
        && name != "format"
        && name != "db_role"
        && name != "settings_list";
}

expected<void> getSettings(ContextPtr ctx, const RequestPtr& request) {
    auto get = [&]() -> expected<MapOptions> {
        try {
            if (request->content.type == "application" && request->content.subtype == "json") {
                const std::string body(request->raw_body.begin(), request->raw_body.end());
                return getSettingsFromBody(body);
            }
            return getSettingsFromParams(request->url.params);
        } catch(const std::exception& e) {
            return make_unexpected(error_code(
                make_error_code(Error::noSuchNode), e.what())
            );
        }
    };
    return get()
        .bind([&](auto&& settings) -> expected<void> {
            if (settings.empty()) {
                return make_unexpected(error_code(
                    make_error_code(Error::noSuchNode), "No parameters to set")
                );
            }
            ctx->settings(std::make_shared<MapOptions>(std::move(settings)));
            return {};
        });
}

expected<MapOptions> getSettingsFromBody(const std::string& body) {
    using yamail::data::deserialization::fromJson;

    MapOptions settings;
    fromJson<MapOptions>(body, settings);
    return settings;
}

expected<MapOptions> getSettingsFromParams(const std::multimap<std::string, std::string>& params) {
    using yamail::data::deserialization::fromJson;

    MapOptions settings;
    auto& single_settings = settings.single_settings;

    auto it = boost::range::find_if(params, [](auto& v) {return !isValidParamName(v.first);});
    if (it != params.end()) {
        return make_unexpected(error_code(
            make_error_code(Error::noSuchNode), "Wrong setting name: "
                R"(")" + it->first + R"(")")
        );
    }
    if (const auto it = params.find("json_data"); it != params.end()) {
        fromJson<MapOptions>(it->second, settings);

        auto ws = boost::range::find_if(single_settings, [](auto& v) {return !isValidParamName(v.first);});
        if (ws != single_settings.end()) {
            return make_unexpected(error_code(
                make_error_code(Error::noSuchNode), "Wrong setting name: "
                    R"(")" + ws->first + R"(")")
                );
        }
    } else if (const auto it = params.find("signs"); it != params.end()) {
        fromJson<SignaturesListOpt>(it->second, settings.signs);
    } else {
        boost::range::for_each(
            params | boost::adaptors::filtered(
                [] (const auto& v) {return isValidSettings(v.first);}
            ),
            [&](auto& v) {single_settings.emplace(v);}
        );
    }
    return settings;
}

expected<void> getUid(ContextPtr ctx, const std::multimap<std::string, std::string>& map) {
    auto it = map.find("uid");
    if (it == map.end()) {
        return make_unexpected(error_code(
            make_error_code(Error::invalidParameterError), R"(parameter 'uid' not found)")
        );
    }
    auto& uid = it->second;
    std::int64_t numericUid;
    if (!boost::conversion::try_lexical_convert<std::int64_t>(uid, numericUid)) {
        return make_unexpected(error_code(
            make_error_code(Error::invalidParameterError), R"(parameter 'uid' value is invalid)")
        );
    }
    if (numericUid <= 0) {
        return make_unexpected(error_code(
            make_error_code(Error::invalidParameterError), R"(parameter 'uid' value is invalid)")
        );
    }
    ctx->uid(uid);
    return {};
}

expected<void> getSettingsListFromParams(ContextPtr ctx, const std::multimap<std::string, std::string>& map) {
    auto it = map.find("settings_list");
    if (it == map.end()) {
        return make_unexpected(error_code(
            make_error_code(Error::invalidParameterError), R"(parameter 'settings_list' not found)")
        );
    }
    SettingsList settingsList;
    boost::split(settingsList, it->second, boost::is_any_of("\r"), boost::token_compress_on);

    if (settingsList.empty() || (settingsList.size() == 1 && settingsList.front() == "")) {
        return make_unexpected(error_code(
            make_error_code(Error::invalidParameterError), R"(no values in 'settings_list' parameter)")
        );
    }
    ctx->settingsList(settingsList);
    return {};
}

expected<void> getAskValidator(ContextPtr ctx, const std::multimap<std::string, std::string>& map) {
    auto it = map.find("ask_validator");
    if (it != map.end()) {
        ctx->askValidator(it->second == "y");
    }
    return {};
}

expected<void> getDatabaseRole(ContextPtr ctx, const std::multimap<std::string, std::string>& map) {
    auto it = map.find("db_role");
    if (it != map.end()) {
        auto& role = it->second;
        if (role == "master") {
            ctx->databaseRole(DatabaseRole::Master);
        } else if (role == "replica") {
            ctx->databaseRole(DatabaseRole::Replica);
        } else {
            return make_unexpected(error_code(
                make_error_code(Error::invalidParameterError), R"('db_role' value is invalid)")
            );
        }
    }
    return {};
}

expected<format::Type> getFormat(const std::multimap<std::string, std::string>& map) {
    auto it = map.find("format");
    if (it != map.end()) {
        if (it->second != "json") {
            return make_unexpected(error_code(
                make_error_code(Error::invalidParameterError), R"(unsupported 'format' value)")
            );
        }
    }
    return format::Type::Json;
}

expected<void> getSettingsListFromBody(ContextPtr ctx, const std::string& body) {
    SettingsList settingsList;
    try {
        yamail::data::deserialization::fromJson<SettingsList>(body, settingsList);
    } catch (const std::exception& e) {
        return make_unexpected(error_code(
            make_error_code(Error::noSuchNode), e.what())
        );
    }
    if (settingsList.empty()) {
        return make_unexpected(error_code(
            make_error_code(Error::invalidParameterError), R"(no values in 'settings_list' parameter)")
        );
    }
    ctx->settingsList(settingsList);
    return {};
}

}
