#include <yamail/data/deserialization/ptree.h>

#include <internal/service/factory.h>
#include <internal/query/query_register_impl.h>

#include <pgg/service/uid_resolver.h>

#include <internal/common/error_code.h>
#include <internal/common/logger.h>
#include <internal/common/profiler.h>

#include <internal/macs/impl.h>
#include <internal/macs/config_reflection.h>
#include <internal/macs/sharpei_httpclient.h>

namespace settings::macs {

inline decltype(auto) getStartegy(ContextPtr ctx) {
    return ctx->databaseRole() == DatabaseRole::Master
        ? ::macs::pg::masterOnly
        : ::macs::pg::readReplicaThenMaster;
}

void Impl::init(const yplatform::ptree& ptree) {
    using yamail::data::deserialization::fromPtree;
    config = fromPtree<settings::Config>(ptree);
    config.sharpei.httpClient = std::make_shared<sharpei::SharpeiHttpClient>(
        yplatform::find<yhttp::call, std::shared_ptr>("http_client"),
        config.sharpei.settings.timeout
    );
    if(config.database.user.empty()) {
        throw std::runtime_error("Database user not found in config");
    }
    mode = config.mode == "corp" ? Mode::corp : Mode::production;

    queryConf = readQueryConfFile(
        config.database.query_conf,
        ::macs::pg::query::Queries(),
        ::macs::pg::query::QueryParameters()
    );
    сonnectionPools.create(config.database);
}


decltype(auto) Impl::repository(ContextPtr ctx) {
    auto repository = ctx->settingsRepository();
    if (repository) {
        return repository;
    }
    auto newRepository = ::macs::pg::ServiceFactoryImpl(
        сonnectionPools.get(),
        createSharpeiUidResolverFactory(config.sharpei),
        {}
    )
    .requestInfo(
        pgg::RequestInfo {ctx->requestId(), ctx->uniq_id(), ctx->clientType(), ctx->userIp()}
    )
    .credentials(
        pgg::Credentials {config.database.user, ""}
    )
    .profiler(pa::create())
    .logger(logging::getPggLogger(ctx->requestId()))
    .queryConf(queryConf)
    .transactionTimeout(
        std::chrono::milliseconds {config.database.query_timeout_ms}
    )
    .autoQueryHandleStartegy(getStartegy(ctx))
    .product(ctx->uid())->settings()
    .setMode(mode);

    ctx->settingsRepository(newRepository);
    return newRepository;

}

expected<bool> Impl::initSettings(ContextPtr ctx) {
    mail_errors::error_code ec;
    auto data = repository(ctx)->initSettings(
        io_result::make_yield_context(ctx->yield())[ec]
    );
    if (ec) {
        return make_unexpected(std::move(ec));
    }
    return data;
}

expected<Settings> Impl::getSettings(ContextPtr ctx, SettingsList settings) {
    mail_errors::error_code ec;
    auto data = repository(ctx)->getSettings(
        std::move(settings),
        io_result::make_yield_context(ctx->yield())[ec]
    );
    if (ec) {
        return make_unexpected(std::move(ec));
    } else if (!data) {
        return make_unexpected<mail_errors::error_code>(make_error_code(Error::getError));
    }
    return *data;
}

expected<Profile> Impl::getProfile(ContextPtr ctx, SettingsList settings) {
    mail_errors::error_code ec;
    auto data = repository(ctx)->getProfile(
        std::move(settings),
        io_result::make_yield_context(ctx->yield())[ec]
    );
    if (ec) {
        return make_unexpected(std::move(ec));
    } else if (!data) {
        return make_unexpected<mail_errors::error_code>(make_error_code(Error::getError));
    }
    return *data;
}

expected<Parameters> Impl::getParameters(ContextPtr ctx, SettingsList settings) {
    mail_errors::error_code ec;
    auto data = repository(ctx)->getParameters(
        std::move(settings),
        io_result::make_yield_context(ctx->yield())[ec]
    );
    if (ec) {
        return make_unexpected(std::move(ec));
    } else if (!data) {
        return make_unexpected<mail_errors::error_code>(make_error_code(Error::getError));
    }
    return *data;
}

expected<bool> Impl::updateParameters(ContextPtr ctx, ParametersPtr parameters) {
    mail_errors::error_code ec;
    auto data = repository(ctx)->updateParameters(
        std::move(parameters),
        io_result::make_yield_context(ctx->yield())[ec]
    );
    if (ec) {
        return make_unexpected(std::move(ec));
    }
    return data;
}

expected<bool> Impl::updateProtectedParameters(ContextPtr ctx, ParametersPtr parameters) {
    mail_errors::error_code ec;
    auto data = repository(ctx)->updateProtectedParameters(
        std::move(parameters),
        io_result::make_yield_context(ctx->yield())[ec]
    );
    if (ec) {
        return make_unexpected(std::move(ec));
    }
    return data;
}

expected<bool> Impl::updateProfile(ContextPtr ctx, ProfilePtr profile) {
    mail_errors::error_code ec;
    auto data = repository(ctx)->updateProfile(
        std::move(profile),
        io_result::make_yield_context(ctx->yield())[ec]
    );
    if (ec) {
        return make_unexpected(std::move(ec));
    }
    return data;
}

expected<bool> Impl::deleteSettings(ContextPtr ctx) {
    mail_errors::error_code ec;
    auto data = repository(ctx)->deleteSettings(
        io_result::make_yield_context(ctx->yield())[ec]
    );
    if (ec) {
        return make_unexpected(std::move(ec));
    }
    return data;
}

expected<bool> Impl::eraseParameters(ContextPtr ctx, SettingsList settings) {
    mail_errors::error_code ec;
    auto data = repository(ctx)->eraseParameters(
        std::move(settings),
        io_result::make_yield_context(ctx->yield())[ec]
    );
    if (ec) {
        return make_unexpected(std::move(ec));
    }
    return data;
}

using impl = Impl;

} //settings::macs

#include <yplatform/module_registration.h>

DEFINE_SERVICE_OBJECT(settings::macs::impl)
