#include <boost/algorithm/string.hpp>

#include <internal/recognizer/UTFizer.h>

namespace settings::utfizer {

const CharsetTraitsMap UTFizer::map;

std::pair<std::int32_t, std::int32_t> UTFizer::recognize(const std::string& text) const {
    return Recognizer::recognize(recognizer, text);
}

Recognizer::CodesEnum UTFizer::recognizeEncoding(const std::string& text) const {
    return Recognizer::recognizeEncoding(recognizer, text);
}

std::string UTFizer::convert(const CharsetTraitsMap::CharsetTraits & traits, const std::string & text) const {
    try {
        std::string out;
        traits.convertor.convert(text, out, traits.charset);
        return out;
    } catch( ... ) {
    }
    return text;
}

std::string UTFizer::convert(const std::string & charset, const std::string & text) const {
    try {
        return convert(map.getCharsetTraits(charset), text);
    } catch( ... ) {
    }
    return convert(recognizeEncoding(text), text);
}

std::string UTFizer::convert(Recognizer::CodesEnum code, const std::string & text) const {
    return convert(map.getCharsetTraits(code), text);
}

bool UTFizer::untrustedCharset( const std::string & charset ) const {
    using boost::algorithm::iequals;
    return iequals(charset,"us-ascii") ||
           iequals(charset,"iso-8859-1") ||
           charset.empty();
}

std::string UTFizer::utfize(const std::string & text) const {
    return convert(recognizeEncoding(text), text);
}

}
