#include <sstream>
#include <stdexcept>

#include <boost/numeric/conversion/cast.hpp>

#include <webmail-recognizer.h>

#include <internal/recognizer/wrapper.h>

namespace Recognizer {

struct NumericCastError : std::logic_error {
    template <class T>
    NumericCastError(T value, const char* function, const char* valueName, const char* error)
        : std::logic_error(makeMessage(value, function, valueName, error)) {}

    template <class T>
    static std::string makeMessage(T value, const char* function, const char* valueName, const char* error) {
        std::ostringstream stream;
        stream << "Bad numeric cast of " << valueName << " which is " << value << " in " << function << ": " << error;
        return stream.str();
    }
};

template <class R, class T>
R numericCast(T value, const char* functionName, const char* valueName) {
    try {
        return boost::numeric_cast<R>(value);
    } catch (const boost::bad_numeric_cast &error) {
        throw NumericCastError(value, functionName, valueName, error.what());
    }
}

} // namespace Recognizer

#define RECOGNIZER_NUMERIC_CAST(Type, value) \
    Recognizer::numericCast<Type>(value, __PRETTY_FUNCTION__, #value)

namespace Recognizer
{

const TWebmailRecognizer& instance() {
    static const TWebmailRecognizer res;
    return res;
}

std::pair<std::int32_t, std::int32_t> recognize(const TWebmailRecognizer& instance, const std::string& text) {
    std::int32_t code = 0;
    std::int32_t lang = 0;
    unsigned int length = RECOGNIZER_NUMERIC_CAST(unsigned int, text.length());
    const Result res(instance.AnalyzeText(text.c_str(), length, code, lang));
    return res.succeeded() ? std::make_pair(
        checkRestraints(code, CODES_UNSUPPORTED, CODES_MAX),
        checkRestraints(lang, LANG_UNK, LANG_MAX)
    ): std::make_pair(CODES_UNKNOWN, LANG_UNK);
}

CodesEnum recognizeEncoding(const TWebmailRecognizer& instance, const std::string& text) {
    int code = 0;
    unsigned int length = RECOGNIZER_NUMERIC_CAST(unsigned int, text.length());
    const Result res(instance.AnalyzeEncoding(text.c_str(), length, code));
    return res.succeeded() ? convertEnum(code, CODES_UNSUPPORTED, CODES_MAX) : CODES_UNSUPPORTED;
}

}
