#include <yamail/data/serialization/json_writer.h>

#include <yplatform/find.h>

#include <user_journal/service_factory.h>
#include <user_journal/parameters/settings.h>

#include <internal/common/error.h>
#include <internal/common/types_reflection.h>

#include <internal/user_journal/impl.h>
#include <internal/user_journal/log.h>

#include <locale>

namespace settings::user_journal {

void Impl::init(const yplatform::ptree& config) {
    const auto tskvFile = config.get<std::string>("tskv_file");

    tskvWriter = std::make_shared<TskvWriter>(tskvFile);

    ::user_journal::ServiceFactory factory;

    service = factory
        .tskv(config.get_child("user_journal"), tskvWriter->interface())
        .logger(boost::make_shared<Log>())
        .locale(std::locale(std::locale("ru_RU.UTF-8"), "C", std::locale::numeric))
        .product();
}

void Impl::reload(const yplatform::ptree& config) {
    const UniqueLock lock(mutex);
    init(config);
}

void Impl::asyncLogSettingsUpdate(ContextPtr ctx, MapOptions value) {
    asyncLogSettingsUpdateImpl(ctx, std::move(value));
}

void Impl::asyncLogSettingsUpdate(ContextPtr ctx, SettingsMap value) {
    asyncLogSettingsUpdateImpl(ctx, std::move(value));
}

void Impl::asyncLogSettingsUpdate(ContextPtr ctx, SignaturesList value) {
    asyncLogSettingsUpdateImpl(ctx, std::move(value));
}

Impl::SettingsVector::SettingsVector(MapOptions&& value)
        : Base(std::make_move_iterator(value.single_settings.begin()),
               std::make_move_iterator(value.single_settings.end())) {
    if (value.signs) {
        emplace_back(Setting("signs", reflect(*value.signs)));
    }
}

Impl::SettingsVector::SettingsVector(const SignaturesList& value)
    : Base({Setting("signs", reflect(value))}) {}

template <class T>
void Impl::asyncLogSettingsUpdateImpl(ContextPtr ctx, T&& value) {
    try {
        logSettingsUpdate(ctx, ctx->uid(), SettingsVector(std::move(value)));
    } catch (...) {
        logCurrentException(ctx);
    }
}

void Impl::logSettingsUpdate(ContextPtr ctx, const std::string& uid, const SettingsVector& values) {
    using namespace ::user_journal::parameters;
    using ::user_journal::Journal;

    const SharedLock lock(mutex);

    auto journal = service->createJournal(uid,
        id::module("settings"),
        id::clientType(ctx->clientType()),
        id::clientVersion(ctx->clientVersion()),
        id::testBuckets(ctx->testBuckets()),
        id::enabledTestBuckets(ctx->enabledTestBuckets()),
        id::ip(ctx->userIp()),
        id::requestId(ctx->requestId())
    );

    for (const auto& value : values) {
        journal.write<SettingsUpdate>(
            id::hidden(true),
            id::mdb(""),
            id::affected(1),
            id::state(value.name + "=" + value.value),
            id::settingName(value.name),
            id::settingValue(value.value)
        );
    }
}

std::string Impl::reflect(const SignaturesList& value) {
    yamail::data::serialization::JsonWriter<SignaturesList> writer(value);
    return writer.result();
}

} // namespace settings::user_journal

#include <yplatform/module_registration.h>

DEFINE_SERVICE_OBJECT(settings::user_journal::Impl)
