from argparse import ArgumentParser
from collections import namedtuple
from mail.unistat.cpp.cython.meters import (
    AccessLogCount,
    AccessLogCountByFirstStatusDigit,
    AccessLogCountByPathAndFirstStatusDigit,
    AccessLogRequestTimeHist,
    AccessLogRequestTimeHistByPath,
    HttpClientHttpRequestTotalTimeHist,
    HttpClientHttpRequestCountByStatus,
    PaLogRequestTimeHist,
    SupervisorLogRestartMeters,
)
from mail.unistat.cpp.cython.count_by_condition import (
    Counter,
    Condition,
    Signal,
    Filter,
    update_signal_names,
)
from mail.settings.unistat.cpp.run import (
    run as run_unistat,
)
from mail.settings.unistat.cpp.errors import (
    ServiceError
)
import os
import yaml
import logging
import mail.unistat.cpp.cython.logs as logs


logging.basicConfig(level=logging.WARNING, format='[%(asctime)s] [%(levelname)s]: %(message)s')
log = logging.getLogger(__name__)


HTTP_HIST_BUCKETS = (0, 20, 100, 300, 1000)


SERVICE_ERROR_CATEGORY = 'settings'
SERVICE_ERROR_SIGNALS = [
    Signal(cond=Condition.Equals(getattr(ServiceError, attr)), name=attr)
    for attr in dir(ServiceError) if not callable(getattr(ServiceError, attr)) and not attr.startswith('__')
] + [
    Signal(cond=Condition.AnyStr(), name='settings_unknown'),
]


CATEGORY_SIGNALS = [
    Signal(cond=Condition.Equals('sharpei_client'),                 name='SharpeiError'),
    Signal(cond=Condition.Equals('apq.pq'),                         name='PostgreSqlError'),
    Signal(cond=Condition.Equals('pgg::error::CommonCategory'),     name='PostgreSqlError'),
    Signal(cond=Condition.Equals('pgg::error::ReadonlyCategory'),   name='PostgreSqlError'),
    Signal(cond=Condition.Equals('ymod_httpclient'),                name='HttpClientError'),
    Signal(cond=Condition.Equals('macs::error::Category'),          name='MacsError'),

    Signal(cond=Condition.AnyStr(),                                 name='unknown'),
]


def make_service_log_meters():
    meters = []
    prefix = 'errors_{}'
    suffix = 'dmmm'
    code_category = 'error_code.category'
    code_value = 'error_code.value'

    signals = update_signal_names(SERVICE_ERROR_SIGNALS, prefix, suffix)
    meters.append(Counter(code_value, signals,  [Filter(key=code_category, cond=Condition.Equals(SERVICE_ERROR_CATEGORY))]))

    signals = update_signal_names(CATEGORY_SIGNALS, prefix, suffix)
    meters.append(Counter(code_category, signals, []))

    return meters


def make_access_log_meters(name_prefix, endpoints):
    meters = [
        AccessLogCount(name_prefix),
        AccessLogCountByFirstStatusDigit(name_prefix),
        AccessLogRequestTimeHist(HTTP_HIST_BUCKETS, 'access_log_request'),
    ]

    for endpoint in endpoints:
        meters += [
            AccessLogCountByPathAndFirstStatusDigit(endpoint, name_prefix),
            AccessLogRequestTimeHistByPath(HTTP_HIST_BUCKETS, endpoint, 'access_log_request'),
        ]

    return meters


def make_supervisor_meters():
    return [
        SupervisorLogRestartMeters('supervisor')
    ]


def parse_args():
    parser = ArgumentParser()
    parser.add_argument('-H', '--host', default='::')
    parser.add_argument('-p', '--port', default=8082, type=int)
    parser.add_argument('-d', '--dir', default='.')
    parser.add_argument('-s', action='store_true', help='read file logs from start')
    parser.add_argument('-l', '--log', default='', help='path for yplatform.log')
    parser.add_argument('--supervisorlog', help='path for supervisord.log')
    parser.add_argument('service_config')
    return parser.parse_args()


def get_module_config(data, name):
    return next(v['configuration'] for v in data['config']['modules']['module'] if v['system']['name'] == name)


def get_logger_sink_path(data, name, sink_index=0):
    return os.path.join(os.curdir, data['config']['log'][name]['sinks'][sink_index]['path'])


ServiceConfig = namedtuple('ServiceConfig', (
    'httpclient_log',
    'access_log',
    'service_log',
    'profiler_log',

    'service_endpoints'
))


def get_service_endpoints(tvm_guard_rules, exclude={'/ping'}):
    endpoints = []
    for rule in tvm_guard_rules:
        endpoints += [path for path in rule['paths'] if path not in exclude]
    return endpoints


def make_service_config(data):
    yset_http = get_module_config(data, 'settings_http')
    return ServiceConfig(
        httpclient_log=get_logger_sink_path(data, 'http_client'),
        access_log=get_logger_sink_path(data, 'access'),
        service_log=get_logger_sink_path(data, 'settings'),
        profiler_log=os.path.join(os.curdir, get_module_config(data, 'settings_pa')['file']),
        service_endpoints=get_service_endpoints(yset_http['tvm_guard']['rules']),
    )


def make_http_client_log_meters(_):
    uri_filter = 'blackbox'
    return [
        HttpClientHttpRequestCountByStatus(uri_filter, 'count_by_status_passport'),
        HttpClientHttpRequestTotalTimeHist(HTTP_HIST_BUCKETS, uri_filter, 'time_hist_passport'),
    ]


def make_pa_log_meters():
    return [
        PaLogRequestTimeHist((0, 10, 20, 30, 40, 50, 100, 500, 1000, 5000), 'postgresql',  'pa'),
    ]


def main():
    args = parse_args()
    log.info('chdir %s' % os.path.abspath(args.dir))
    os.chdir(args.dir)

    with open(args.service_config) as f:
        config = make_service_config(yaml.load(f, Loader=yaml.FullLoader))

    fast_forward = args.s

    service_log = logs.TskvLog([], make_service_log_meters(), fast_forward, config.service_log)
    access_log = logs.AccessTskv([], make_access_log_meters('access_log', config.service_endpoints), fast_forward, config.access_log)
    http_client_log = logs.HttpClientLog([], make_http_client_log_meters(config), fast_forward, config.httpclient_log)
    pa_log = logs.PaLog([], make_pa_log_meters(), fast_forward, config.profiler_log)

    logs_list = [service_log, access_log, http_client_log, pa_log]

    if args.supervisorlog is not None:
        supervisor_log = logs.SupervisorLog(
            [],
            make_supervisor_meters(),
            fast_forward,
            args.supervisorlog
        )
        logs_list.append(supervisor_log)

    run_unistat(args.host, args.port, logs_list, args.log, logLevel='info')


if __name__ == '__main__':
    main()
