#pragma once

#include <unordered_map>
#include <boost/circular_buffer.hpp>
#include <internal/shard.h>
#include <internal/guarded.h>

namespace sharpei {
namespace cache {

class LimitedCounter {
public:
    LimitedCounter(std::size_t max, std::size_t value = 0);
    LimitedCounter& operator +=(std::size_t value);
    LimitedCounter& operator -=(std::size_t value);
    std::size_t value() const;

private:
    std::size_t max_;
    std::size_t value_;
};

class Availability {
public:
    using Status = Shard::Database::Status;
    using History = std::vector<Status>;
    using GetMethod = Status ((Availability::*)() const);
    using UpdateMethod = void (Availability::*)();

    Availability(std::size_t historyCapacity = 1, std::size_t errorsLimit = 1);

    Status get(GetMethod method) const;
    Status smooth() const;
    Status recent() const;

    History getHistory() const;

    void update(UpdateMethod method);
    void alive();
    void dead();

private:
    std::size_t errorsLimit_;
    LimitedCounter errorsCount_;
    boost::circular_buffer<bool> responseHistory_;
};

class StatusCache {
public:
    using Status = Shard::Database::Status;
    using OptStatus = boost::optional<Status>;
    using Address = Shard::Database::Address;
    using History = Availability::History;

    StatusCache(std::size_t historyCapacity, std::size_t errorsLimit);

    std::size_t historyCapacity() const;
    std::size_t errorsLimit() const;

    OptStatus get(Shard::Id shardId, const Address& address, Availability::GetMethod method) const;
    History getHistory(Shard::Id shardId, const Address& address) const;
    void alive(Shard::Id shardId, const Address& address);
    void dead(Shard::Id shardId, const Address& address);
    void update(Shard::Id shardId, const Address& address, Availability::UpdateMethod method);
    void erase(Shard::Id shardId);

private:
    using Hosts = std::map<Address, Availability>;
    using Shards = std::unordered_map<Shard::Id, Hosts>;

    const std::size_t historyCapacity_;
    const std::size_t errorsLimit_;
    guarded<Shards, std::shared_mutex> shards;
};

} // namespace cache
} // namespace sharpei
