#pragma once

#include <internal/async_operation.h>
#include <internal/config.h>
#include <internal/db/adaptors/meta_master_provider.h>
#include <internal/db/adaptors/repl_mon_based_meta_master_provider.h>
#include <internal/db/adaptors/request_executor.h>
#include <internal/db/pools/meta_pool.h>
#include <internal/db/query.h>
#include <internal/db/shard_weight.h>
#include <internal/domain.h>
#include <internal/errors.h>
#include <internal/helpers.h>
#include <internal/logger.h>
#include <internal/organization.h>
#include <internal/shard.h>

#include <type_traits>

namespace sharpei::db {

enum class UserType {
    existing,
    deleted
};

enum class QueryType {
    withData,
    withoutData
};

struct RegData {
    WeightedShardIds weightedShardIds;
    boost::optional<Shard::Id> userShardId;
};

struct BaseMetaAdaptor {
    using GetShardHandler = std::function<void (const ShardWithoutRoles&)>;
    using GetAllShardsHandler = std::function<void (const std::vector<ShardWithoutRoles>&)>;
    using GetShardIdHandler = std::function<void(Shard::Id)>;
    using GetRegDataHandler = std::function<void(RegData&)>;
    using GetUserDataHandler = std::function<void(Shard::Id, boost::optional<std::string>)>;
    using GetMasterHandler = std::function<void(std::string)>;
    using ErrorHandler = std::function<void (const ExplainedError&)>;
    using FinishHandler = std::function<void()>;

    virtual ~BaseMetaAdaptor() = default;

    virtual void getShard(Shard::Id shardId, const GetShardHandler& handler, const ErrorHandler& errorHandler) const = 0;
    virtual void getAllShards(const GetAllShardsHandler& handler, const ErrorHandler& errorHandler) const = 0;
    virtual void ping(const FinishHandler& handler, const ErrorHandler& errorHandler) const = 0;
    virtual void getMaster(const GetMasterHandler& handler,
                           const ErrorHandler& errorHandler) const = 0;
    virtual void getRegData(GetRegDataHandler handler, ErrorHandler errorHandler) const = 0;
    virtual void getDomainShardId(const DomainId domainId, GetShardIdHandler handler,
        ErrorHandler errorHandler) const = 0;
    virtual void getOrganizationShardId(const OrgId orgId, GetShardIdHandler handler,
        ErrorHandler errorHandler) const = 0;
};

using BaseMetaAdaptorPtr = std::shared_ptr<const BaseMetaAdaptor>;

template <class UserIdValue>
class MetaAdaptor : public BaseMetaAdaptor {
public:
    using UserId = BasicUserId<UserIdValue>;

    virtual void getUserRegData(const UserId& uid, GetRegDataHandler handler, ErrorHandler errorHandler) const = 0;
    virtual void getUserData(const UserId& uid, const GetUserDataHandler& handler, const ErrorHandler& errorHandler, QueryType queryType = QueryType::withData) const = 0;
    virtual void getDeletedUserData(const UserId& uid, const GetUserDataHandler& handler, const ErrorHandler& errorHandler, QueryType queryType = QueryType::withData) const = 0;
};

template <class T>
using MetaAdaptorPtr = std::shared_ptr<const MetaAdaptor<T>>;

namespace detail {

using GetShardHandler = BaseMetaAdaptor::GetShardHandler;
using GetAllShardsHandler = BaseMetaAdaptor::GetAllShardsHandler;
using GetShardIdHandler = BaseMetaAdaptor::GetShardIdHandler;
using GetRegDataHandler = BaseMetaAdaptor::GetRegDataHandler;
using GetUserDataHandler = BaseMetaAdaptor::GetUserDataHandler;
using GetMasterHandler = BaseMetaAdaptor::GetMasterHandler;
using ErrorHandler = BaseMetaAdaptor::ErrorHandler;
using FinishHandler = BaseMetaAdaptor::FinishHandler;


template <class Id>
struct GetRegDataQuery {};

// clang-format off
template <>
struct GetRegDataQuery<std::monostate> {
    static constexpr auto value =
        "SELECT shard_id, reg_weight, NULL AS user_shard_id "
        "FROM shards.shards";
};

template <>
struct GetRegDataQuery<BasicUserId<std::int64_t>> {
    static constexpr auto value =
        "SELECT shard_id, reg_weight, "
            "(SELECT shard_id AS user_shard_id FROM shards.users WHERE uid = $1) "
        "FROM shards.shards";
};

template <>
struct GetRegDataQuery<BasicUserId<std::string>> {
    static constexpr auto value =
        "SELECT shard_id, reg_weight, "
            "(SELECT shard_id AS user_shard_id FROM shards.text_users WHERE uid = $1) "
        "FROM shards.shards";
};
// clang-format on

template <class Id>
class GetRegDataExecutor: public RequestExecutor
                        , public EnableStaticConstructor<GetRegDataExecutor<Id>> {
private:
    friend class EnableStaticConstructor<GetRegDataExecutor>;

    GetRegDataExecutor(const AdaptorConfig& config,
                       const MetaPoolPtr& pool,
                       const Scribe& scribe,
                       const Id& id,
                       GetRegDataHandler handler,
                       ErrorHandler errorHandler)
        : RequestExecutor(config, pool, scribe, errorHandler), id_(id), handler_(handler) {
    }

    apq::query makeQuery() const override final {
        apq::query query(GetRegDataQuery<Id>::value);
        db::bind(query, id_);
        return query;
    }

    std::string getProfilingId() const override final {
        return "get_reg_data";
    }

    void onEmptyResult() override final {
        errorHandler_(ExplainedError(Error::metaRequestError, "empty result"));
    }

    void onCorrectResult(apq::row_iterator it) override final {
        try {
            RegData regData;

            for (; it != apq::row_iterator(); ++it) {
                const auto shardId = extractField<Shard::Id>(it, "shard_id");
                const auto regWeight = extractField<ShardWeight>(it, "reg_weight");
                regData.weightedShardIds.emplace_back(shardId, regWeight);
                regData.userShardId = extractField<boost::optional<Shard::Id>>(it, "user_shard_id");
            }

            handler_(regData);
        } catch (const std::exception& e) {
            errorHandler_(ExplainedError(Error::metaRequestError,
                "error when handle request to meta db: " + std::string(e.what())));
        }
    }

    const Id id_;
    const GetRegDataHandler handler_;
};

template <AnyOf<std::string, std::int64_t> UserIdValue, UserType userType = UserType::existing>
std::string makeGetUserDataQuery(QueryType queryType = QueryType::withData) {
    std::string table;
    if constexpr (std::is_same_v<std::string, UserIdValue>) {
        table = "shards.text_users";
    } else if constexpr (userType == UserType::existing) {
        table = "shards.users";
    } else {
        table = "shards.deleted_users";
    }

    std::string fields;
    if (queryType == QueryType::withData) {
        fields = "shard_id, data::json";
    } else {
        fields = "shard_id";
    }

    return fmt::format("SELECT {} "
                       "FROM {} "
                       "WHERE uid = $1", fields, table);
}

template <class Id, UserType userType = UserType::existing>
class GetUserDataExecutor: public RequestExecutor
                         , public EnableStaticConstructor<GetUserDataExecutor<Id, userType>> {
private:
    friend class EnableStaticConstructor<GetUserDataExecutor>;

    GetUserDataExecutor(const AdaptorConfig& config, const MetaPoolPtr& pool, const Scribe& scribe,
                        const Id& uid, const GetUserDataHandler& handler, const ErrorHandler& errorHandler, QueryType queryType = QueryType::withData)
        : RequestExecutor(config, pool, scribe, errorHandler)
        , uid_(uid)
        , handler_(handler)
        , queryType_(queryType)
    {}

    apq::query makeQuery() const override final {
        apq::query query(makeGetUserDataQuery<typename Id::Value, userType>(queryType_));
        db::bind(query, uid_);
        return query;
    }

    std::string getProfilingId() const override final {
        return "get_user_data";
    }

    void onEmptyResult() override final {
        const auto uid = sharpei::to_string(uid_);
        static const auto where = std::string{"get_"} + (userType == UserType::existing ? "existing" : "deleted") + "_user_data";
        LOGDOG_(scribe_.logger, warning, log::where_name=where, log::error_code=Error::uidNotFound, log::uid=uid);
        errorHandler_(ExplainedError(Error::uidNotFound, "error in request to meta db: uid=" + uid + " not found"));
    }

    void onCorrectResult(apq::row_iterator it) override final {
        try {
            const auto shardId = extractField<Shard::Id>(it, "shard_id");
            boost::optional<std::string> data;
            if (queryType_ == QueryType::withData) {
                if (!it->is_null(1)) {
                    std::string value;
                    it->at(1, value);
                    data = std::move(value);
                }
            }
            handler_(shardId, std::move(data));
        } catch (const std::exception& e) {
            errorHandler_(ExplainedError(Error::metaRequestError,
                "error in request to meta db: " + std::string(e.what())));
        }
    }

    Id uid_;
    GetUserDataHandler handler_;
    QueryType queryType_;
};

class BaseMetaAdaptorImpl: public BaseMetaAdaptor {
public:
    BaseMetaAdaptorImpl(const AdaptorConfig& config,
                        const MetaPoolPtr& pool,
                        const Scribe& scribe,
                        MetaMasterProviderPtr metaMasterProvider);

    void getShard(Shard::Id shardId, const GetShardHandler& handler, const ErrorHandler& errorHandler) const override final;
    void getAllShards(const GetAllShardsHandler& handler, const ErrorHandler& errorHandler) const override final;
    void ping(const FinishHandler& handler, const ErrorHandler& errorHandler) const override final;
    void getMaster(const GetMasterHandler& handler,
                   const ErrorHandler& errorHandler) const override final;
    void getRegData(GetRegDataHandler handler, ErrorHandler errorHandler) const override final;
    void getDomainShardId(const DomainId domainId, GetShardIdHandler handler,
        ErrorHandler errorHandler) const override final;
    void getOrganizationShardId(const OrgId orgId, GetShardIdHandler handler,
        ErrorHandler errorHandler) const override final;

private:
    const AdaptorConfig config_;
    const MetaPoolPtr pool_;
    const Scribe scribe_;
    const MetaMasterProviderPtr metaMasterProvider_;
};

template <template <class, auto ...> class Executor, class UserIdValue, auto ... params>
class TryCastUidToInt64 {
public:
    template <class ... Args>
    static void execute(const AdaptorConfig& config, const MetaPoolPtr& pool, const Scribe& scribe,
            const BasicUserId<UserIdValue>& uid, Args&& ... args) {
        bool casted;
        std::int64_t value;
        std::tie(casted, value) = toInt64(uid.value());
        if (casted) {
            Executor<BasicUserId<std::int64_t>, params ...>::create(
                config, pool, scribe, BasicUserId<std::int64_t>(value), std::forward<Args>(args) ...
            )->execute();
        } else {
            Executor<BasicUserId<UserIdValue>, params ...>::create(
                config, pool, scribe, uid, std::forward<Args>(args) ...
            )->execute();
        }
    }

private:
    static std::pair<bool, std::int64_t> toInt64(const std::string& value) {
        return lexicalCast<std::int64_t>(value);
    }

    static std::pair<bool, std::int64_t> toInt64(std::int64_t value) {
        return {true, value};
    }
};

template <class UserIdValue>
class MetaAdaptorImpl: public MetaAdaptor<UserIdValue> {
public:
    using UserId = typename MetaAdaptor<UserIdValue>::UserId;

    MetaAdaptorImpl(const BaseMetaAdaptorPtr& basic,
                    const AdaptorConfig& config,
                    const MetaPoolPtr& pool,
                    const Scribe& scribe)
        : basic_(basic), config_(config), pool_(pool), scribe_(scribe) {
    }

    void getShard(Shard::Id shardId,
                  const GetShardHandler& handler,
                  const ErrorHandler& errorHandler) const override final {
        basic_->getShard(shardId, handler, errorHandler);
    }

    void getAllShards(const GetAllShardsHandler& handler,
                      const ErrorHandler& errorHandler) const override final {
        basic_->getAllShards(handler, errorHandler);
    }

    void getUserRegData(const UserId& uid,
                    GetRegDataHandler handler,
                    ErrorHandler errorHandler) const override final {
        TryCastUidToInt64<GetRegDataExecutor, typename UserId::Value>::execute(
            config_, pool_, scribe_, uid, handler, errorHandler);
    }

    void ping(const FinishHandler& handler,
              const ErrorHandler& errorHandler) const override final {
        basic_->ping(handler, errorHandler);
    }

    void getUserData(const UserId& uid,
                     const GetUserDataHandler& handler,
                     const ErrorHandler& errorHandler,
                     QueryType queryType) const override final {
        TryCastUidToInt64<GetUserDataExecutor, typename UserId::Value, UserType::existing>::execute(
                    config_, pool_, scribe_, uid, handler, errorHandler, queryType);
    }

    void getDeletedUserData(const UserId& uid,
                            const GetUserDataHandler& handler,
                            const ErrorHandler& errorHandler,
                            QueryType queryType) const override final {
        TryCastUidToInt64<GetUserDataExecutor, typename UserId::Value, UserType::deleted>::execute(
                    config_, pool_, scribe_, uid, handler, errorHandler, queryType);
    }

    void getMaster(const GetMasterHandler& handler,
                   const ErrorHandler& errorHandler) const override final {
        basic_->getMaster(handler, errorHandler);
    }

    void getRegData(GetRegDataHandler handler, ErrorHandler errorHandler) const override final {
        basic_->getRegData(std::move(handler), std::move(errorHandler));
    }

    void getDomainShardId(const DomainId domainId, GetShardIdHandler handler,
            ErrorHandler errorHandler) const override final {
        basic_->getDomainShardId(domainId, std::move(handler), std::move(errorHandler));
    }

    void getOrganizationShardId(const OrgId orgId, GetShardIdHandler handler,
            ErrorHandler errorHandler) const override final {
        basic_->getOrganizationShardId(orgId, std::move(handler), std::move(errorHandler));
    }

private:
    BaseMetaAdaptorPtr basic_;
    AdaptorConfig config_;
    MetaPoolPtr pool_;
    Scribe scribe_;
};

} // namespace detail

namespace coro {

class BaseMetaAdaptor {
public:
    BaseMetaAdaptor(BaseMetaAdaptorPtr impl) : impl(std::move(impl)) {}

    template <class CompletionToken>
    auto getMaster(CompletionToken&& token) const {
        return performAsyncOperation<std::string, SplitHandlersAsyncOperation>(
            std::forward<CompletionToken>(token), [&](auto onValue, auto onError) {
                return impl->getMaster(std::move(onValue), std::move(onError));
            });
    }

    template <class CompletionToken>
    auto getRegData(CompletionToken&& token) const {
        return performAsyncOperation<RegData, SplitHandlersAsyncOperation>(
            std::forward<CompletionToken>(token),
            [&](auto onValue, auto onError) { return impl->getRegData(std::move(onValue), std::move(onError)); });
    }

    template <class CompletionToken>
    auto getDomainShardId(const DomainId domainId, CompletionToken&& token) const {
        return performAsyncOperation<Shard::Id, SplitHandlersAsyncOperation>(
            std::forward<CompletionToken>(token),
            [&] (auto onValue, auto onError) {
                return impl->getDomainShardId(domainId, std::move(onValue), std::move(onError));
            }
        );
    }

    template <class CompletionToken>
    auto getOrganizationShardId(const OrgId orgId, CompletionToken&& token) const {
        return performAsyncOperation<Shard::Id, SplitHandlersAsyncOperation>(
            std::forward<CompletionToken>(token),
            [&] (auto onValue, auto onError) {
                return impl->getOrganizationShardId(orgId, std::move(onValue), std::move(onError));
            }
        );
    }

private:
    BaseMetaAdaptorPtr impl;
};

} // namespace coro
} // namespace sharpei::db
