#pragma once

#include <internal/config.h>
#include <internal/errors.h>
#include <internal/logger.h>
#include <internal/shard.h>
#include <internal/user.h>
#include <internal/async_profile.h>
#include <internal/domain.h>
#include <internal/organization.h>
#include <internal/async_operation.h>
#include <internal/db/query.h>
#include <internal/db/pools/connection_pool.h>

namespace sharpei::db {

using PeersPoolPtr = std::shared_ptr<IConnectionPool>;

template <class UserIdValue>
struct CreateUserParams {
    BasicUserId<UserIdValue> uid;
    Shard::Id shardId;
};

template <class UserIdValue>
inline void bind(apq::query& query, const CreateUserParams<UserIdValue>& value) {
    db::bind(query, value.uid);
    query.bind_const_int64(value.shardId);
}

struct CreateDomainParams {
    DomainId domainId;
    Shard::Id shardId;
};

inline void bind(apq::query& query, const CreateDomainParams& value) {
    query.bind_const_int64(value.domainId);
    query.bind_const_int64(value.shardId);
}

struct CreateOrganizationParams {
    OrgId orgId;
    std::optional<DomainId> domainId;
    Shard::Id shardId;
};

inline void bind(apq::query& query, const CreateOrganizationParams& value) {
    query.bind_const_int64(value.orgId);
    if (value.domainId) {
        query.bind_const_int64(*value.domainId);
    } else {
        query.bind_null();
    }
    query.bind_const_int64(value.shardId);
}

template <class UserIdValue>
struct UpdateUserParams {
    BasicUserId<UserIdValue> uid;
    boost::optional<Shard::Id> shardId;
    boost::optional<Shard::Id> newShardId;
    boost::optional<std::string> data;
};

template <class UserIdValue>
class PeersAdaptor {
public:
    using CreateUserParams = db::CreateUserParams<UserIdValue>;
    using UpdateUserParams = db::UpdateUserParams<UserIdValue>;

    PeersAdaptor(const AdaptorConfig& config, Scribe scribe, PeersPoolPtr pool)
        : config_(config),
          scribe_(std::move(scribe)),
          pool_(std::move(pool)) {
    }

    template <class Handler>
    void createUser(const std::string& master, const CreateUserParams& params, Handler&& handler) const;

    template <class Handler>
    void updateUser(const std::string& master, UpdateUserParams params, Handler&& handler) const;

private:
    AdaptorConfig config_;
    Scribe scribe_;
    PeersPoolPtr pool_;

    ConnectionInfo makeConnectionInfo(const std::string& metaMasterHost) const {
        return ConnectionInfo(config_.connInfoWOHost, metaMasterHost);
    }
};

template <class Tag>
class Performer {
public:
    Performer(ProfilerPtr profiler, PeersPoolPtr pool, ConnectionInfo connInfo)
            : profiler_(std::move(profiler)),
              pool_(std::move(pool)),
              connInfo_(std::move(connInfo)) {
    }

    template <class Handler>
    void perform(const apq::query& query, apq::time_traits::duration_type timeout, Handler&& handler) const {
        using ApqHandler = apq::connection_pool::request_handler_t;
        Profile profile(profiler_, Tag::operation, connInfo_.host());
        const ApqHandler profiled = asyncProfile(std::move(profile), std::forward<Handler>(handler));
        pool_->get(connInfo_)->async_request(query, profiled, apq::result_format_binary, timeout);
    }

private:
    ProfilerPtr profiler_;
    PeersPoolPtr pool_;
    ConnectionInfo connInfo_;
};

template <class UserIdValue>
struct CreateUserQuery {};

template <>
struct CreateUserQuery<std::int64_t> {
    static constexpr auto value = "SELECT code.create_user_2($1::bigint, $2::integer)";
};

template <>
struct CreateUserQuery<std::string> {
    static constexpr auto value = "SELECT code.create_user_2($1::text, $2::integer)";
};

template <class UserIdValue>
struct CreateUserTraits {
    using Query = CreateUserQuery<UserIdValue>;

    struct Tag {
        static constexpr auto operation = "create_user";
    };
};

template <class Params, class Traits, class Handler>
class Create : public std::enable_shared_from_this<Create<Params, Traits, Handler>> {
public:
    Create(const Params& params, ProfilerPtr profiler, PeersPoolPtr pool,
               ConnectionInfo connInfo, Handler&& handler)
            : performer_(std::move(profiler), std::move(pool), std::move(connInfo)),
              handler_(std::forward<Handler>(handler)),
              query_(Traits::Query::value) {
        db::bind(query_, params);
    }

    void perform(apq::time_traits::duration_type timeout) {
        auto self = this->shared_from_this();
        auto handler = [self] (auto ... args) { self->handle(std::move(args) ...); };
        performer_.perform(query_, timeout, std::move(handler));
    }

private:
    Performer<typename Traits::Tag> performer_;
    Handler handler_;
    apq::query query_;

    void handle(apq::result result, apq::row_iterator it) {
        if (result.code()) {
            handler_(ExplainedError(chooseMetaRequestErrorCode(result), makeErrorMessage(query_, result.message())), Shard::Id());
        } else if (it == apq::row_iterator()) {
            handler_(ExplainedError(Error::metaRequestError, makeErrorMessage(query_, "empty result")), Shard::Id());
        } else if (it->size() == 0) {
            handler_(ExplainedError(Error::metaRequestError, makeErrorMessage(query_, "no columns in rows")), Shard::Id());
        } else if (it->is_null(0)) {
            handler_(ExplainedError(Error::metaRequestError, makeErrorMessage(query_, "result is null")), Shard::Id());
        } else {
            Shard::Id result;
            it->at(0, result);
            handler_(ExplainedError(Error::ok), result);
        }
    }
};

template <class Handler, class UserIdValue>
using CreateUser = Create<
    db::CreateUserParams<UserIdValue>,
    CreateUserTraits<UserIdValue>,
    Handler
>;

template <class Handler, class UserIdValue>
auto makeCreateUser(const CreateUserParams<UserIdValue>& params, ProfilerPtr profiler, PeersPoolPtr pool,
        ConnectionInfo connInfo, Handler&& handler) {
    return std::make_shared<CreateUser<Handler, UserIdValue>>(params, std::move(profiler), std::move(pool),
        std::move(connInfo), std::forward<Handler>(handler));
}

template <class UserIdValue>
template <class Handler>
void PeersAdaptor<UserIdValue>::createUser(const std::string& master, const CreateUserParams& params, Handler&& handler) const {
    auto connInfo = makeConnectionInfo(master);
    const auto performer = makeCreateUser(params, scribe_.profiler, pool_,
        std::move(connInfo), std::forward<Handler>(handler));
    performer->perform(config_.requestTimeout);
}

template <class UserIdValue>
struct UpdateUserQueries {};

template <>
struct UpdateUserQueries<std::int64_t> {
    static constexpr auto uidAndDataQuery = "SELECT code.update_user($1::bigint /* uid */, $2::jsonb /* data */)";
    static constexpr auto uidAndShardIdQuery = "/* uid, shard_id */";
    static constexpr auto uidAndShardIdAndDataQuery = "SELECT code.update_user($1::bigint /* uid */, $2::integer /* shard_id */, $3::jsonb /* data */)";
    static constexpr auto uidAndShardIdAndNewShardIdQuery = "SELECT code.update_user($1::bigint /* uid */, $2::integer /* shard_id */, $3::integer /* new_shard_id */)";
    static constexpr auto uidAndShardIdAndNewShardIdAndDataQuery = "SELECT code.update_user($1::bigint /* uid */, $2::integer /* shard_id */, $3::integer /* new_shard_id */, $4::jsonb /* data */)";
};

template <>
struct UpdateUserQueries<std::string> {
    static constexpr auto uidAndDataQuery = "SELECT code.update_user($1::text /* uid */, $2::jsonb /* data */)";
    static constexpr auto uidAndShardIdQuery = "/* uid, shard_id */";
    static constexpr auto uidAndShardIdAndDataQuery = "SELECT code.update_user($1::text /* uid */, $2::integer /* shard_id */, $3::jsonb /* data */)";
    static constexpr auto uidAndShardIdAndNewShardIdQuery = "SELECT code.update_user($1::text /* uid */, $2::integer /* shard_id */, $3::integer /* new_shard_id */)";
    static constexpr auto uidAndShardIdAndNewShardIdAndDataQuery = "SELECT code.update_user($1::text /* uid */, $2::integer /* shard_id */, $3::integer /* new_shard_id */, $4::jsonb /* data */)";
};

struct UpdateUserTraits {
    template <class UserIdValue>
    struct QuerySelector {
        using Queries = UpdateUserQueries<UserIdValue>;

        void onNewShardId() {
            if (current_ == Queries::uidAndShardIdQuery) {
                current_ = Queries::uidAndShardIdAndNewShardIdQuery;
                return;
            }

            onInvalidTransition(__func__);
        }

        void onShardId() {
            if (current_ == nullptr) {
                current_ = Queries::uidAndShardIdQuery;
                return;
            }

            onInvalidTransition(__func__);
        }

        void onData() {
            if (current_ == nullptr) {
                current_ = Queries::uidAndDataQuery;
                return;
            } else if (current_ == Queries::uidAndShardIdQuery) {
                current_ = Queries::uidAndShardIdAndDataQuery;
                return;
            } else if (current_ == Queries::uidAndShardIdAndNewShardIdQuery) {
                current_ = Queries::uidAndShardIdAndNewShardIdAndDataQuery;
                return;
            }

            onInvalidTransition(__func__);
        }

        const char* current() const {
            return current_;
        }

        bool isFinal() const {
            return current() == Queries::uidAndDataQuery
                    || current() == Queries::uidAndShardIdAndDataQuery
                    || current() == Queries::uidAndShardIdAndNewShardIdQuery
                    || current() == Queries::uidAndShardIdAndNewShardIdAndDataQuery;
        }

    private:
        const char* current_ = nullptr;

        void onInvalidTransition(const char* symbol) {
            throw std::logic_error(fmt::format("invalid transition {} when current state is '{}'", symbol, current()));
        }
    };

    struct Tag {
        static constexpr auto operation = "update_user";
    };
};

template <class Handler, class UserIdValue>
class UpdateUser : public std::enable_shared_from_this<UpdateUser<Handler, UserIdValue>> {
public:
    using Params = db::UpdateUserParams<UserIdValue>;

    UpdateUser(Params&& params, ProfilerPtr profiler, PeersPoolPtr pool,
               ConnectionInfo connInfo, Handler&& handler)
            : performer_(std::move(profiler), std::move(pool), std::move(connInfo)),
              handler_(std::forward<Handler>(handler)),
              query_(std::string()) {
        db::bind(query_, params.uid);
        UpdateUserTraits::QuerySelector<UserIdValue> selector;
        if (params.shardId) {
            selector.onShardId();
            query_.bind_const_int64(params.shardId.get());
        }
        if (params.newShardId) {
            selector.onNewShardId();
            query_.bind_const_int64(params.newShardId.get());
        }
        if (params.data) {
            selector.onData();
            query_.bind_const_string(std::move(params.data.get()));
        }
        if (!selector.isFinal()) {
            throw std::logic_error("invalid combination of initialized parameters: suitable query not found");
        }
        query_.text_ = selector.current();
    }

    void perform(apq::time_traits::duration_type timeout) {
        auto self = this->shared_from_this();
        auto handler = [self] (auto ... args) { self->handle(std::move(args) ...); };
        performer_.perform(query_, timeout, std::move(handler));
    }

private:
    Performer<UpdateUserTraits::Tag> performer_;
    Handler handler_;
    apq::query query_;

    void handle(apq::result result, apq::row_iterator it) {
        if (result.code()) {
            handler_(ExplainedError(chooseMetaRequestErrorCode(result), makeErrorMessage(query_, result.message())));
        } else if (it == apq::row_iterator()) {
            handler_(ExplainedError(Error::metaRequestError, makeErrorMessage(query_, "empty result")));
        } else if (it->size() == 0) {
            handler_(ExplainedError(Error::metaRequestError, makeErrorMessage(query_, "no columns in rows")));
        } else if (it->is_null(0)) {
            handler_(ExplainedError(Error::metaRequestError, makeErrorMessage(query_, "result is null")));
        } else {
            std::string result;
            it->at(0, result);
            if (result == "success") {
                handler_(ExplainedError(Error::ok));
            } else if (result == "user_not_found") {
                handler_(ExplainedError(Error::uidNotFound));
            } else if (result == "invalid_user_shard_id") {
                handler_(ExplainedError(Error::invalidUserShardId));
            } else {
                handler_(ExplainedError(Error::metaRequestError, makeErrorMessage(query_, "unhandled request result: " + result)));
            }
        }
    }
};

template <class Handler, class UserIdValue>
auto makeUpdateUser(UpdateUserParams<UserIdValue>&& params, ProfilerPtr profiler, PeersPoolPtr pool,
        ConnectionInfo connInfo, Handler&& handler) {
    return std::make_shared<UpdateUser<Handler, UserIdValue>>(std::move(params), std::move(profiler), std::move(pool),
        std::move(connInfo), std::forward<Handler>(handler));
}

template <class UserIdValue>
template <class Handler>
void PeersAdaptor<UserIdValue>::updateUser(const std::string& master, UpdateUserParams params, Handler&& handler) const {
    auto connInfo = makeConnectionInfo(master);
    const auto performer = makeUpdateUser(std::move(params), scribe_.profiler, pool_,
        std::move(connInfo), std::forward<Handler>(handler));
    performer->perform(config_.requestTimeout);
}

struct CreateDomainQuery {
    static constexpr auto value = "SELECT code.create_domain($1::bigint, $2::integer)";
};

struct CreateDomainTraits {
    using Query = CreateDomainQuery;

    struct Tag {
        static constexpr auto operation = "create_domain";
    };
};

template <class Handler>
using CreateDomain = Create<
    CreateDomainParams,
    CreateDomainTraits,
    Handler
>;

template <class Handler>
auto makeCreateDomain(const CreateDomainParams& params, ProfilerPtr profiler, PeersPoolPtr pool,
        ConnectionInfo connInfo, Handler&& handler) {
    return std::make_shared<CreateDomain<Handler>>(params, std::move(profiler), std::move(pool),
        std::move(connInfo), std::forward<Handler>(handler));
}

struct CreateOrganizationQuery {
    static constexpr auto value = R"(
        SELECT code.create_organization(
            i_org_id := $1::bigint,
            i_domain_id := $2::bigint,
            i_shard_id := $3::integer
        )
    )";
};

struct CreateOrganizationTraits {
    using Query = CreateOrganizationQuery;

    struct Tag {
        static constexpr auto operation = "create_organization";
    };
};

template <class Handler>
using CreateOrganization = Create<
    CreateOrganizationParams,
    CreateOrganizationTraits,
    Handler
>;

template <class Handler>
auto makeCreateOrganization(const CreateOrganizationParams& params, ProfilerPtr profiler, PeersPoolPtr pool,
        ConnectionInfo connInfo, Handler&& handler) {
    return std::make_shared<CreateOrganization<Handler>>(params, std::move(profiler), std::move(pool),
        std::move(connInfo), std::forward<Handler>(handler));
}

class BasePeersAdaptor {
public:
    BasePeersAdaptor(const AdaptorConfig& config, Scribe scribe, PeersPoolPtr pool)
        : config_(config),
          scribe_(std::move(scribe)),
          pool_(std::move(pool)) {
    }

    template <class CompletionToken>
    auto createDomain(const std::string& master, const CreateDomainParams& params,
            CompletionToken&& token) const {
        return performAsyncOperation<Shard::Id, SingleHandlerAsyncOperation>(
            std::forward<CompletionToken>(token),
            [&] (auto handler) {
                const auto performer = makeCreateDomain(params, scribe_.profiler, pool_,
                    makeConnectionInfo(master), std::move(handler));
                performer->perform(config_.requestTimeout);
            }
        );
    }

    template <class CompletionToken>
    auto createOrganization(const std::string& master, const CreateOrganizationParams& params,
            CompletionToken&& token) const {
        return performAsyncOperation<Shard::Id, SingleHandlerAsyncOperation>(
            std::forward<CompletionToken>(token),
            [&] (auto handler) {
                const auto performer = makeCreateOrganization(params, scribe_.profiler, pool_,
                    makeConnectionInfo(master), std::move(handler));
                performer->perform(config_.requestTimeout);
            }
        );
    }

private:
    AdaptorConfig config_;
    Scribe scribe_;
    PeersPoolPtr pool_;

    ConnectionInfo makeConnectionInfo(const std::string& metaMasterHost) const {
        return ConnectionInfo(config_.connInfoWOHost, metaMasterHost);
    }
};

} // namespace sharpei::db
