#pragma once

#include <internal/config.h>
#include <internal/db/adaptors/meta_master_provider.h>
#include <internal/db/adaptors/request_executor.h>
#include <internal/db/pools/meta_pool.h>
#include <internal/logger.h>

namespace sharpei::db {

namespace detail {

class GetMasterExecutor : public RequestExecutor, public EnableStaticConstructor<GetMasterExecutor> {
private:
    friend class EnableStaticConstructor<GetMasterExecutor>;

    using GetMasterHandler = std::function<void(std::string)>;

    GetMasterExecutor(const AdaptorConfig& config,
                      const MetaPoolPtr& pool,
                      const Scribe& scribe,
                      const GetMasterHandler& handler,
                      const ErrorHandler& errorHandler)
        : RequestExecutor(config, pool, scribe, errorHandler), handler_(handler) {
    }

    apq::query makeQuery() const override final {
        static constexpr char queryStr[] = "SELECT master FROM repl_mon";
        return apq::query(queryStr);
    }

    std::string getProfilingId() const override final {
        return "get_master";
    }

    void onEmptyResult() override final {
        errorHandler_(ExplainedError(Error::metaRequestError, "no result"));
    }

    void onCorrectResult(apq::row_iterator it) override final {
        try {
            handler_(extractField<std::string>(it, "master"));
        } catch (const std::exception& e) {
            errorHandler_(ExplainedError(Error::metaRequestError, makeErrorMessage(*query_, e.what())));
        }
    }

    GetMasterHandler handler_;
};

}  // namespace detail

class ReplMonBasedMetaMasterProvider : public IMetaMasterProvider {
public:
    ReplMonBasedMetaMasterProvider(const AdaptorConfig& config, const MetaPoolPtr& pool, const Scribe& scribe)
        : config_(config), pool_(pool), scribe_(scribe) {
    }

    void getMaster(const GetMasterHandler& handler, const ErrorHandler& errorHandler) const {
        auto onError = [logger = scribe_.logger, errorHandler](const auto error) {
            LOGDOG_(logger,
                    error,
                    log::where_name = "repl_mon_based_meta_master_provider",
                    log::error_code = Error::metaMasterProviderError,
                    log::reason = error);
            errorHandler(error);
        };
        auto executor = detail::GetMasterExecutor::create(config_, pool_, scribe_, handler, onError);
        executor->execute();
    }

private:
    const AdaptorConfig config_;
    const MetaPoolPtr pool_;
    const Scribe scribe_;
};

}  // namespace sharpei::db
