#pragma once

#include <boost/lexical_cast.hpp>

#include <optional>

#include <spdlog/details/format.h>

#define MEMBER_ACCESSORS(type, myClass, field) \
public:                                        \
    myClass& field(std::optional<type> f) {    \
        field##_ = std::move(f);               \
        return *this;                          \
    }                                          \
    const std::optional<type>& field() const { \
        return field##_;                       \
    }

#define DECL_MEMBER_WITH_ACCESSORS(type, myClass, field) \
private:                                                 \
    std::optional<type> field##_;                        \
    MEMBER_ACCESSORS(type, myClass, field)

namespace sharpei::db {

struct AuthInfo {
    DECL_MEMBER_WITH_ACCESSORS(std::string, AuthInfo, password)
    DECL_MEMBER_WITH_ACCESSORS(std::string, AuthInfo, user)
    DECL_MEMBER_WITH_ACCESSORS(std::string, AuthInfo, sslmode)
};

struct ConnectionInfoWOHost {
    // meant to be used only by config deserialization code
    ConnectionInfoWOHost() = default;

    ConnectionInfoWOHost(const std::string& dbname, unsigned port, const AuthInfo& authInfo)
    : authInfo(authInfo), dbname(dbname), port(port) {
    }

    AuthInfo authInfo;
    std::string dbname;
    unsigned port;
};

class ConnectionInfo {
public:
    ConnectionInfo(const ConnectionInfoWOHost& base, const std::string& host) : base_(base), host_(host) {
    }

    ConnectionInfo(const std::string& host, unsigned port, const std::string& dbname, const AuthInfo& authInfo)
    : base_(dbname, port, authInfo), host_(host) {
    }

    const std::string& host() const {
        return host_;
    }

    unsigned port() const {
        return base_.port;
    }

    const std::string& dbname() const {
        return base_.dbname;
    }

    const std::optional<std::string>& user() const {
        return base_.authInfo.user();
    }

    const std::optional<std::string>& password() const {
        return base_.authInfo.password();
    }

    const std::optional<std::string>& sslmode() const {
        return base_.authInfo.sslmode();
    }

    std::string toString() const {
        std::string res = fmt::format("host={} port={} dbname={}", host(), std::to_string(port()), dbname());
        if (user()) {
            res += " user=" + user().value();
        }
        if (password()) {
            res += " password=" + password().value();
        }
        if (sslmode()) {
            res += " sslmode=" + sslmode().value();
        }
        return res;
    }

private:
    constexpr auto makeTuple() const {
        return std::forward_as_tuple(host(), base_.port, dbname(), user(), password(), sslmode());
    }

public:
    bool operator<(const ConnectionInfo& other) const {
        return makeTuple() < other.makeTuple();
    }

    bool operator!=(const ConnectionInfo& other) const {
        return makeTuple() != other.makeTuple();
    }

private:
    const ConnectionInfoWOHost base_;
    const std::string host_;
};

}  // namespace sharpei::db

#undef MEMBER_ACCESSORS
#undef DECL_MEMBER_WITH_ACCESSORS
