#pragma once

#include <internal/cache/cache.h>
#include <internal/db/conn_info.h>
#include <mail/sharpei/include/internal/db/filtration_strategy.h>
#include <internal/db/ep/endpoint_provider.h>
#include <internal/dc_vanga_config.h>
#include <internal/logger.h>

#include <yamail/expected.h>

#include <memory>
#include <optional>

namespace sharpei::db {

struct IEndpointSelectionStrategy {
    virtual ~IEndpointSelectionStrategy() = default;

    virtual yamail::expected<Shard::Database> selectEndpoint(const std::vector<Shard::Database>& dbs) = 0;
};

using EndpointSelectionStrategyPtr = std::shared_ptr<IEndpointSelectionStrategy>;

class SimpleEndpointSelectionStrategy : public IEndpointSelectionStrategy {
public:
    yamail::expected<Shard::Database> selectEndpoint(const std::vector<Shard::Database>& dbs) override;

private:
    inline static thread_local unsigned iterator = 0;
};

class DCAwareMainStrategy : public IEndpointSelectionStrategy {
public:
    DCAwareMainStrategy(const std::string& localDC, std::size_t aliveHostsThreshold);

    DCAwareMainStrategy(const dc_vanga::Rules& rules, std::size_t aliveHostsThreshold);

    yamail::expected<Shard::Database> selectEndpoint(const std::vector<Shard::Database>& dbs) override;

private:
    const std::string localDC_;
    const std::size_t aliveHostsThreshold_;
    SimpleEndpointSelectionStrategy impl_;
};

IEndpointProvider::Endpoint toEndpoint(const Shard::Database& db, const AuthInfo& authInfo);

class MetaCacheBasedEndpointProvider : public IEndpointProvider {
public:
    MetaCacheBasedEndpointProvider(cache::CachePtr metaCache, const db::AuthInfo& authInfo,
                                   FiltrationStrategyPtr filter, EndpointSelectionStrategyPtr mainStrategy,
                                   EndpointSelectionStrategyPtr fallbackStrategy);

    Endpoint getNext() override;

private:
    std::vector<Shard::Database> getMetaDatabases() const;

    const cache::CachePtr metaCache_;
    const db::AuthInfo authInfo_;
    const FiltrationStrategyPtr filter_;
    const EndpointSelectionStrategyPtr mainStrategy_;
    const EndpointSelectionStrategyPtr fallbackStrategy_;
};

EndpointProviderPtr makeMetaCacheBasedEndpointProvider(cache::CachePtr metaCache,
                                                       const db::AuthInfo& authInfo,
                                                       const dc_vanga::Rules& rules,
                                                       std::size_t aliveHostsThreshold);
}  // namespace sharpei::db
