#pragma once

#include <internal/db/config.h>
#include <internal/db/conn_info.h>
#include <internal/db/pools/apq_connection_pool.h>
#include <apq/connection_pool.hpp>
#include <apq/dns.hpp>

#include <condition_variable>
#include <thread>

namespace sharpei {
namespace db {

class IConnectionPool {
public:
    typedef std::shared_ptr<ApqConnectionPool> ConnectionPoolPtr;
    typedef std::map<ConnectionInfo, ApqConnectionPool::Stats> Stats;

    virtual ~IConnectionPool() = default;
    virtual ConnectionPoolPtr get(const ConnectionInfo& connInfo) = 0;
    virtual Stats stats() = 0;
};

class ConnectionPool: public IConnectionPool {
private:
    typedef apq::time_traits::duration_type Duration;
    typedef boost::asio::io_service Ios;
    typedef std::map<ConnectionInfo, ConnectionPoolPtr> ConnectionPools;
    typedef std::set<ConnectionInfo> CreatingConnections;
    typedef std::unique_lock<std::mutex> Lock;

public:
    ConnectionPool(Ios& ios, const PoolConfig& poolCfg) : ios_(ios), poolCfg_(poolCfg) {
        if (poolCfg_.asyncResolve)
            apq::setup_dns(ios_, apq::dns_settings{.cache_ttl = poolCfg_.dns.cacheTtl});
    }

    ConnectionPoolPtr get(const ConnectionInfo& connInfo) override {
        Lock lock(mutex_);
        ConnectionPools::iterator iter = connectionPools_.find(connInfo);

        while (iter == connectionPools_.end()) {
            if (inProgress(connInfo)) {
                iter = wait(lock, connInfo);
            } else {
                iter = create(lock, connInfo);
                return iter->second;
            }
        }

        return iter->second;
    }

    Stats stats() override {
        Stats result;
        const Lock lock(mutex_);
        for (const auto& pool : connectionPools_) {
            result.insert({pool.first, pool.second->stats()});
        }
        return result;
    }

private:
    bool inProgress(const ConnectionInfo& connInfo) const {
        return creating_.count(connInfo);
    }

    ConnectionPools::iterator wait(Lock& l, const ConnectionInfo& connInfo) {
        cond_.wait(l);
        return connectionPools_.find(connInfo);
    }

    ConnectionPools::iterator create(Lock& l, const ConnectionInfo& connInfo) {
        creating_.insert(connInfo);
        l.unlock();

        auto pool = createPool(connInfo);

        l.lock();
        creating_.erase(connInfo);
        ConnectionPools::iterator i = connectionPools_.insert(std::make_pair(connInfo, pool)).first;
        l.unlock();

        cond_.notify_all();
        return i;
    }

    ConnectionPoolPtr createPool(const ConnectionInfo& connInfo) const {
        ConnectionPoolPtr pool{new ConnectionPoolPtr::element_type{ios_}};
        pool->set_conninfo(connInfo.toString());
        pool->set_connect_timeout(poolCfg_.connectTimeout);
        pool->set_queue_timeout(poolCfg_.queueTimeout);
        pool->set_idle_timeout(poolCfg_.idleTimeout);
        pool->set_limit(poolCfg_.maxConnections);
        pool->set_async_resolve(poolCfg_.asyncResolve);
        pool->set_ipv6_only(poolCfg_.ipv6Only);
        return pool;
    }

    Ios& ios_;
    PoolConfig poolCfg_;
    ConnectionPools connectionPools_;
    CreatingConnections creating_;
    std::mutex mutex_;
    std::condition_variable cond_;
};

inline std::shared_ptr<IConnectionPool> makeConnectionPool(const PoolConfig& config, boost::asio::io_context& io) {
    return std::make_shared<ConnectionPool>(io, config);
}

} // namespace db
} // namespace sharpei
