#pragma once

#include <internal/async_profile.h>
#include <internal/db/config.h>
#include <internal/db/pools/connection_pool.h>
#include <internal/db/ep/endpoint_provider.h>
#include <internal/db/pools/thread_connection_pool_map.h>

namespace sharpei::db {

class MetaPool {
public:
    using ProfilingId = std::string;
    using OptProfilingCtx = std::optional<std::pair<ProfilerPtr, ProfilingId>>;
    using RequestHandler = apq::connection_pool::request_handler_t;

    explicit MetaPool(std::shared_ptr<IConnectionPool> pool, std::shared_ptr<IEndpointProvider> endpointProvider)
    : pool_(std::move(pool)), endpointProvider_(std::move(endpointProvider)) {
    }

    void async_request(const apq::query& query, RequestHandler handler, apq::result_format rt,
                       apq::time_traits::duration_type tm, OptProfilingCtx&& profilingCtx = std::nullopt) {
        const auto& endpoint = endpointProvider_->getNext();
        auto connection = pool_->get(endpoint);
        auto h = getHandler(std::move(handler), std::move(profilingCtx), endpoint.host());
        connection->async_request(query, std::move(h), rt, tm);
    }

    void async_update(const apq::query& query, apq::connection_pool::update_handler_t handler,
                      apq::time_traits::duration_type tm) {
        const auto& endpoint = endpointProvider_->getNext();
        auto connection = pool_->get(endpoint);
        connection->async_update(query, std::move(handler), tm);
    }

    auto stats() {
        return pool_->stats();
    }

private:
    RequestHandler getHandler(RequestHandler&& handler, OptProfilingCtx&& profilingCtx, const std::string& host) {
        if (profilingCtx.has_value()) {
            auto&& [profiler, profilingId] = std::move(profilingCtx).value();
            Profile profile(std::move(profiler), std::move(profilingId), host);
            return asyncProfile(std::move(profile), std::move(handler));
        } else {
            return handler;
        }
    }

    std::shared_ptr<IConnectionPool> pool_;
    std::shared_ptr<IEndpointProvider> endpointProvider_;
};

using MetaPoolPtr = std::shared_ptr<MetaPool>;

inline MetaPoolPtr makeMetaPool(const PoolConfig& config, std::shared_ptr<IEndpointProvider> endpointProvider) {
    ThreadConnectionPoolMap::Services services;
    for (const auto& pool : yplatform::global_net_reactor->get_pools()) {
        services.push_back(pool->io());
    }
    auto pool = std::make_shared<ThreadConnectionPoolMap>(services, config);
    return std::make_shared<MetaPool>(std::move(pool), std::move(endpointProvider));
}

}  // namespace sharpei::db
