#pragma once

#include <internal/dc_vanga_config.h>
#include <yamail/data/reflection/reflection.h>

#include <boost/xpressive/xpressive.hpp>

#include <algorithm>
#include <iterator>
#include <optional>
#include <regex>
#include <stdexcept>
#include <string>
#include <unordered_map>

#include <unistd.h>

namespace sharpei::dc_vanga {

inline std::string getHostName() {
    char buffer[1024] = {};
    if (auto ec = gethostname(buffer, sizeof(buffer)); !ec) {
        return buffer;
    } else {
        char err[1024] = {};
        strerror_r(ec, err, sizeof(err));
        throw std::runtime_error("gethostname returned error: " + std::string(err));
    }
}

struct Rule {
    using Regex = boost::xpressive::sregex;
    using Mapping = std::unordered_map<std::string, std::string>;

    Regex rg;
    Mapping mapping;  // dcid -> dc
};

inline std::vector<Rule> toRules(const std::vector<RuleConfig>& rules) {
    std::vector<Rule> res;
    res.reserve(rules.size());
    const auto adaptMapping = [](const std::vector<Mapping>& mapping) {
        Rule::Mapping res;
        for (const auto& m : mapping) {
            res[m.dcId] = yamail::data::reflection::to_string(m.dc);
        }
        return res;
    };
    std::transform(rules.begin(), rules.end(), std::back_inserter(res), [&](const RuleConfig& rule) {
        return Rule{.rg = boost::xpressive::sregex::compile(rule.regex), .mapping = adaptMapping(rule.mapping)};
    });
    return res;
}

inline std::optional<std::string> getHostDC(const std::string& hostname, const std::vector<Rule>& rules) {
    std::optional<std::string> dc;
    for (const auto& rule : rules) {
        boost::xpressive::smatch match;
        if (boost::xpressive::regex_match(hostname, match, rule.rg)) {
            std::string dcId;
            try {
                dcId = match["dcid"];
            } catch (const boost::xpressive::regex_error& e) {
                throw std::logic_error("no named capture group with name 'dcid'");
            }
            if (dc.has_value()) {
                throw std::logic_error("hostname matches more than one regex");
            }
            if (const auto it = rule.mapping.find(dcId); it != rule.mapping.end()) {
                dc = it->second;
            } else {
                throw std::logic_error("no mapping found for the matched dcId");
            }
        }
    }
    return dc;
}

class DCVanga {
public:
    DCVanga(const std::vector<RuleConfig>& rules) : rules_(toRules(rules)) {
    }

    std::string getHostDC(const std::string& hostname) const {
        const auto dc = ::sharpei::dc_vanga::getHostDC(hostname, rules_);
        if (!dc) {
            throw std::logic_error("unable to determine dc for the hostname");
        }
        return *dc;
    }

    std::string getLocalhostDC() const {
        return getHostDC(getHostName());
    }

private:
    const std::vector<Rule> rules_;
};

}  // namespace sharpei::dc_vanga
