#pragma once

#include <apq/error.hpp>
#include <apq/result.hpp>
#include <mail_errors/error_code.h>
#include <yamail/data/reflection/reflection.h>

#include <boost/asio/error.hpp>

#include <stdexcept>
#include <system_error>

namespace sharpei {

struct ParamsException: public std::runtime_error {
    ParamsException(const std::string& msg)
        : std::runtime_error(msg)
    {}
};

struct ConfigException: public std::runtime_error {
    ConfigException(const std::string& msg)
        : std::runtime_error(msg)
    {}
};

struct PgException: public std::runtime_error {
    PgException(const std::string& msg)
        : std::runtime_error(msg)
    {}
};

enum class Error {
    ok = 0,
    internalError,
    uidNotFound,
    shardNotFound,
    appropriateHostNotFound,
    resetError,
    metaRequestError,
    metaRequestTimeout,
    metaRequestApqQueueTimeout,
    modeRolesNotFound,
    invalidRequest,
    unknown,
    noShardWithAliveMaster,
    blackBoxHttpError,
    blackBoxTempError,
    blackBoxFatalError,
    blackBoxParseError,
    mailRegistrationException,
    noCachedShardName,
    noCachedShardDatabasesRoles,
    noCachedShardDatabases,
    invalidUserShardId,
    notPgUser,
    domainNotFound,
    organizationNotFound,
    invalidHttpUrl,
    httpError,
    readJsonWebTokenError,
    iamHttpError,
    iamParseError,
    ycHttpError,
    ycParseError,
    emptyYcHostsCache,
    clusterPollingError,
    clusterPollingResetCacheError,
    metaPollingError,
    metaPollingResetCacheError,
    shardsPollingError,
    shardsPollingResetCacheError,
    endpointProviderError,
    metaMasterProviderError
};

class ErrorCategory : public boost::system::error_category {
public:
    virtual const char* name() const noexcept {
        return "sharpei";
    }

    // Be careful updating these strings: unistat relies on their values.
    virtual std::string message(int ev) const {
        switch (Error(ev)) {
            case Error::ok:
                return "ok";
            case Error::internalError:
                return "internal error";
            case Error::uidNotFound:
                return "uid not found";
            case Error::shardNotFound:
                return "shard not found";
            case Error::appropriateHostNotFound:
                return "appropriate host not found";
            case Error::resetError:
                return "reset error";
            case Error::metaRequestError:
                return "error in request to meta database";
            case Error::metaRequestTimeout:
                return "meta database request timeout";
            case Error::metaRequestApqQueueTimeout:
                return "meta database request expired in apq queue";
            case Error::modeRolesNotFound:
                return "roles for mode not found";
            case Error::invalidRequest:
                return "invalid request";
            case Error::unknown:
                return "unknown error";
            case Error::noShardWithAliveMaster:
                return "shard with alive master not found";
            case Error::blackBoxHttpError:
                return "blackbox http error";
            case Error::blackBoxParseError:
                return "blackbox parse error";
            case Error::blackBoxFatalError:
                return "blackbox fatal error";
            case Error::blackBoxTempError:
                return "blackbox temp error";
            case Error::mailRegistrationException:
                return "exception in mail registration";
            case Error::noCachedShardName:
                return "cached shard name not found";
            case Error::noCachedShardDatabasesRoles:
                return "cached shard databases roles not found";
            case Error::noCachedShardDatabases:
                return "cached shard databases not found";
            case Error::invalidUserShardId:
                return "invalid user shard id";
            case Error::notPgUser:
                return "user is not pg";
            case Error::domainNotFound:
                return "domain not found";
            case Error::organizationNotFound:
                return "organization not found";
            case Error::invalidHttpUrl:
                return "invalid http url";
            case Error::httpError:
                return "http error";
            case Error::readJsonWebTokenError:
                return "read json web token error";
            case Error::iamHttpError:
                return "iam http error";
            case Error::iamParseError:
                return "failed to parse iam response";
            case Error::ycHttpError:
                return "yc http error";
            case Error::ycParseError:
                return "failed to parse yc response";
            case Error::emptyYcHostsCache:
                return "yc hosts cache is empty";
            case Error::clusterPollingError:
                return "cluster polling error";
            case Error::clusterPollingResetCacheError:
                return "cluster polling reset cache error";
            case Error::metaPollingError:
                return "meta polling error";
            case Error::metaPollingResetCacheError:
                return "meta polling reset cache error";
            case Error::shardsPollingError:
                return "shards polling error";
            case Error::shardsPollingResetCacheError:
                return "shards polling reset cache error";
            case Error::endpointProviderError:
                return "endpoint provider error";
            case Error::metaMasterProviderError:
                return "meta master provider error";
        }
        return "sharpei error";
    }

    static const ErrorCategory& instance() {
        static ErrorCategory value;
        return value;
    }
};

inline boost::system::error_code make_error_code(Error e) {
    return boost::system::error_code(static_cast<int>(e), ErrorCategory::instance());
}


enum class RegistrationError {
    ok = 0,
    sharddbRegistrationError,
    maildbRegistrationError,
    registrationInProgress,
    userAlreadyRegistered,
    shardIsOccupiedByUser,
    mdbAndSharddbPrepareFailed,
    illegalParticipantsState
};

class RegistrationErrorCategory : public boost::system::error_category {
public:
    virtual const char* name() const noexcept {
        return "sharpei_registration";
    }

    // Be careful updating these strings: unistat relies on their values.
    virtual std::string message(int ev) const {
        switch (RegistrationError(ev)) {
            case RegistrationError::ok:
                return "successfully registered";
            case RegistrationError::userAlreadyRegistered:
                return "user already registered";
            case RegistrationError::registrationInProgress:
                return "user registration already in progress";
            case RegistrationError::sharddbRegistrationError:
                return "sharddb error during registration";
            case RegistrationError::maildbRegistrationError:
                return "maildb error during registration";
            case RegistrationError::shardIsOccupiedByUser:
                return "shard is occupied by user";
            case RegistrationError::mdbAndSharddbPrepareFailed:
                return "both mdb and sharddb prepare failed";
            case RegistrationError::illegalParticipantsState:
                return "illegal mdb and sharddb state";
        }
        return "sharpei registration error";
    }

    static const RegistrationErrorCategory& instance() {
        static RegistrationErrorCategory value;
        return value;
    }
};

inline boost::system::error_code make_error_code(RegistrationError e) {
    return boost::system::error_code(static_cast<int>(e), RegistrationErrorCategory::instance());
}



using ExplainedError = mail_errors::error_code;

struct ExplainedResponse {
    std::string result;
    std::string description;

    ExplainedResponse()
    {}

    ExplainedResponse(const ExplainedError& error)
        : result(error.base().message())
        , description(error.message())
    {}
};

inline Error chooseMetaRequestErrorCode(const apq::result& res) {
    if (res.code() == boost::asio::error::make_error_code(boost::asio::error::basic_errors::operation_aborted)) {
        return Error::metaRequestTimeout;
    } else if (res.code() == apq::error::make_error_code(apq::error::request_queue_timed_out)) {
        return Error::metaRequestApqQueueTimeout;
    }
    return Error::metaRequestError;
}

} // namespace sharpei

namespace boost {
namespace system {

template<>
struct is_error_code_enum<sharpei::Error> : public std::true_type {};

template<>
struct is_error_code_enum<sharpei::RegistrationError> : public std::true_type {};

} // namespace system
} // namespace boost

BOOST_FUSION_ADAPT_STRUCT(sharpei::ExplainedResponse,
    (std::string, result)
    (std::string, description)
)

YREFLECTION_ADAPT_ADT(sharpei::ExplainedError,
    (result,      std::string, std::string, obj.base().message(), (void)obj; (void)val )
    (description, std::string, std::string, obj.message(),        (void)obj; (void)val )
)
