#ifndef SHARPEI_HELPERS_H
#define SHARPEI_HELPERS_H

#include <memory>
#include <atomic>
#include <vector>
#include <mutex>
#include <charconv>
#include <type_traits>
#include <boost/lexical_cast.hpp>
#include <apq/row_iterator.hpp>
#include <internal/errors.h>

namespace sharpei {

class Spinlock {
public:
    void lock() {
        while (!try_lock()) {
            ;
        }
    }

    bool try_lock() {
        return !lock_.test_and_set(std::memory_order_acquire);
    }

    void unlock() {
        lock_.clear(std::memory_order_release);
    }

private:
    std::atomic_flag lock_ = ATOMIC_FLAG_INIT;
};

template <class Int = unsigned>
class Countdown {
public:
    using Callback = std::function<void()>;

    Countdown(Int initial, Callback cb) : counter_(initial), cb_(std::move(cb)) {
        if (initial <= 0) {
            throw std::logic_error("initial counter value should be positive");
        }
    }

    void decrease() {
        const auto prev = counter_.fetch_sub(1 /*, std::memory_order_acq_rel*/);
        if (prev == 1) {
            cb_();
        } else if (prev == 0) {
            throw std::logic_error("counter decreased beyond zero");
        }
    }

private:
    std::atomic<Int> counter_;
    Callback cb_;
};

template <typename T>
class EnableStaticConstructor {
public:
    typedef std::shared_ptr<T> Ptr;

    template <typename... Args>
    static Ptr create(Args&&... args) {
        // purpose of this struct is not to make friends with make_shared()
        struct Derived: public T { Derived(Args&&... args) :T(std::forward<Args>(args)...) {} };
        return std::make_shared<Derived>(std::forward<Args>(args)...);
    }
};

template <typename T>
class MakeSharedFromThis: public std::enable_shared_from_this<T>, public EnableStaticConstructor<T> {
};

template <typename T, typename RowIterator = apq::row_iterator>
inline T extractField(RowIterator it, const std::string& field) {
    try {
        T result {};
        it->at(field, result);
        return result;
    } catch (const std::exception& e) {
        throw PgException("can't extract field " + field + " from pg result: " + e.what());
    }
}

template <class T>
std::pair<bool, T> lexicalCast(const std::string& str) {
    if constexpr (std::is_same_v<T, std::string>) {
        return {true, str};
    } else {
        T result;
        auto [ptr, ec] = std::from_chars(std::to_address(str.begin()), std::to_address(str.end()), result);
        return {ec == std::errc() && ptr == std::to_address(str.end()), std::move(result)};
    }
    
}

template <class T, class U>
concept SameAs = std::is_same_v<T, U>;

template<typename T, typename ... U>
concept AnyOf = (SameAs<T, U> || ...);
 

} // namespace sharpei

#endif // SHARPEI_HELPERS_H
