#pragma once

#include <mail/apq/include/apq/result.hpp>
#include <mail/logdog/include/logdog/attributes/mail_attributes.h>
#include <mail/logdog/include/logdog/backend/yplatform_log.h>
#include <mail/logdog/include/logdog/format/tskv.h>
#include <mail/logdog/include/logdog/level.h>
#include <mail/logdog/include/logdog/logger.h>
#include <yplatform/log.h>

#include <boost/uuid/uuid.hpp>
#include <boost/uuid/uuid_generators.hpp>

#include <chrono>
#include <exception>
#include <memory>
#include <sstream>
#include <variant>

namespace sharpei {

namespace log {

using namespace logdog::attr;

inline const std::string sharpeiLogKey = "sharpei";

LOGDOG_DEFINE_ATTRIBUTE(std::string, body)
LOGDOG_DEFINE_ATTRIBUTE(std::string, dbname)
LOGDOG_DEFINE_ATTRIBUTE(std::string, host)
LOGDOG_DEFINE_ATTRIBUTE(std::string, method)
LOGDOG_DEFINE_ATTRIBUTE(unsigned int, port)
LOGDOG_DEFINE_ATTRIBUTE(unsigned int, shard_id)
LOGDOG_DEFINE_ATTRIBUTE(std::string, url)
LOGDOG_DEFINE_ATTRIBUTE(std::string, session_id)
LOGDOG_DEFINE_ATTRIBUTE(std::string, transaction_id)
LOGDOG_DEFINE_ATTRIBUTE(apq::result, apq_result)

LOGDOG_DEFINE_ATTRIBUTE(std::string, shard_name)
LOGDOG_DEFINE_ATTRIBUTE(std::string, role)
LOGDOG_DEFINE_ATTRIBUTE(std::string, lag)
LOGDOG_DEFINE_ATTRIBUTE(std::string, smooth_state)
LOGDOG_DEFINE_ATTRIBUTE(std::string, states)

using reason_variant = boost::variant<std::error_code,
                                      boost::system::error_code,
                                      mail_errors::error_code,
                                      std::string,
                                      std::reference_wrapper<const std::exception>>;
LOGDOG_DEFINE_ATTRIBUTE(reason_variant, reason)

// We use custom formatter to be able to set tskv_format in config
struct Formatter {
    template <typename Sequence>
    std::string operator() (const Sequence& args) const {
        using namespace logdog;
        static_assert(hana::Sequence<Sequence>::value, "args should model Hana.Sequence");
        std::ostringstream s;
        auto tskv = ytskv::utf_value(s);
        hana::for_each(args, tskv::attributes_visitor<decltype(tskv)>{tskv});
        tskv << ytskv::endl;
        return s.str();
    }
};

inline auto GetLogger(const std::string& logKey) {
    static constexpr Formatter formatter;
    return logdog::make_log(
        formatter, std::make_shared<yplatform::log::source>(YGLOBAL_LOG_SERVICE, logKey));
}

inline auto GetLogger(const std::string& logKey, const std::string& sessionId, const std::string requestId) {
    return logdog::bind(GetLogger(logKey), session_id=sessionId, request_id=requestId);
}

}

class Profiler {
public:
    virtual ~Profiler() {}
    virtual void write(const std::string& operation, std::chrono::milliseconds duration,
        const std::string& host = "") = 0;
};

typedef std::shared_ptr<Profiler> ProfilerPtr;

struct Scribe {
    using Logger = decltype(log::GetLogger(
        std::declval<const std::string&>(),
        std::declval<const std::string&>(),
        std::declval<const std::string&>()));

    Scribe(ProfilerPtr profiler);
    Scribe(const std::string& sessionId, const std::string& requestId);
    Scribe(const std::string& sessionId, const std::string& requestId, ProfilerPtr profiler);
    Logger logger;
    ProfilerPtr profiler;
    Logger status;
};

std::string generateUniqId();
std::string generateRequestId(boost::uuids::random_generator& generator);

class Profile {
public:
    Profile(ProfilerPtr profiler, std::string operation, std::string host = std::string())
            : profiler(std::move(profiler)),
              operation(std::move(operation)),
              host(std::move(host)) {}

    void operator ()() const {
        const auto now = std::chrono::steady_clock::now();
        const auto duration = std::chrono::duration_cast<std::chrono::milliseconds>(now - begin);
        profiler->write(operation, duration, host);
    }

private:
    std::chrono::steady_clock::time_point begin = std::chrono::steady_clock::now();
    ProfilerPtr profiler;
    std::string operation;
    std::string host;
};

} // namespace sharpei

namespace logdog::tskv {

template <>
struct to_tskv_impl<apq::result> {
    template <typename Out>
    static void apply(Out& out, const char* key, const apq::result& res) {
        const std::string name = key;
        out << ytskv::attr(name + ".sqlstate", res.sqlstate())
            << ytskv::attr(name + ".sqlcode", apq::error::sqlstate::name(res.sqlstate_code()));
    }
};

}
