#pragma once

#include <functional>
#include <internal/mail/db/adaptors/reg_adaptor.h>
#include <internal/helpers.h>
#include <internal/once.h>
#include <internal/db/query.h>
#include <internal/async_profile.h>
#include <boost/format.hpp>
#include <utility>
#include <optional>

namespace sharpei {
namespace mail {
namespace db {

typedef std::shared_ptr<::sharpei::db::ApqConnectionPool> ConnectionPoolSharedPtr;
typedef RegAdaptor::Handler Handler;

namespace detail {
inline auto wrapHandler(Handler handler) {
    return [handler] (auto log, ExplainedError error) {
        log();
        handler(error);
    };
} 
} // namespace detail

typedef decltype(detail::wrapHandler(std::declval<Handler>())) WrappedHandler;
typedef OnceAction<WrappedHandler> OnceHandler;
typedef std::shared_ptr<OnceHandler> OnceHandlerPtr;


template <class Logger>
inline void callOnceError(Logger logger,
                     OnceHandlerPtr handler,
                     const std::string& msg, 
                     RegistrationError error = RegistrationError::ok, 
                     std::optional<std::size_t> uid = std::nullopt) {
    auto log = [logger, uid, error]() {
        if (uid) {
            LOGDOG_(logger, error, log::where_name="register", log::error_code=ExplainedError(error), log::uid=std::to_string(*uid));
        } else {
            LOGDOG_(logger, error, log::where_name="register", log::error_code=ExplainedError(error));
        }
    };

    (*handler)(log, ExplainedError(error, msg));
}

template <class Logger>
inline void callOnce(Logger logger,
                     OnceHandlerPtr handler,
                     const std::string& msg, 
                     RegistrationError error = RegistrationError::ok, 
                     std::optional<std::size_t> uid = std::nullopt) {
    auto log = [logger, uid, error]() {
        if (uid) {
            LOGDOG_(logger, notice, log::where_name="register", log::error_code=ExplainedError(error), log::uid=std::to_string(*uid));
        } else {
            LOGDOG_(logger, notice, log::where_name="register", log::error_code=ExplainedError(error));
        }
    };
    (*handler)(log, ExplainedError(error, msg));
}

enum class ParticipantState {
    Init,
    PrepareInProgress,
    PrepareSuccess,
    PrepareError,
    CommitInProgress,
    CommitSuccess,
    CommitError
};

struct BeginQueryTag {
    static constexpr auto name() { return "register_begin"; }
    static constexpr auto text() { return "BEGIN"; }
};

struct PerepareQueryTag {
    static constexpr auto name() { return "register_prepare"; }
    static constexpr auto text() { return "PREPARE TRANSACTION '%s'"; }
};

struct CommitPreparedQueryTag {
    static constexpr auto name() { return "register_commit_prepared"; }
    static constexpr auto text() { return "COMMIT PREPARED '%s'"; }
};

struct RollbackPreparedQueryTag {
    static constexpr auto name() { return "register_rollback_prepared"; }
    static constexpr auto text() { return "ROLLBACK PREPARED '%s'"; }
};

struct RollbackQueryTag {
    static constexpr auto name() { return "register_rollback"; }
    static constexpr auto text() { return "ROLLBACK"; }
};

struct RegisterUserMaildbQueryTag {
    static constexpr auto name() { return "register_user_maildb"; }
    static constexpr auto text() { return "SELECT code.register_user($1, $2, $3, $4, ($5, $6), $7)"; }
};

struct RegisterUserSharddbQueryTag {
    static constexpr auto name() { return "register_user_sharddb"; }
    static constexpr auto text() { return "SELECT code.register_user($1, $2)"; }
};

template <class TagT>
struct Query {
    using Tag = TagT;
    std::unique_ptr<apq::query> query = std::make_unique<apq::query>(Tag::text());

    template <class Arg>
    static Query format(const Arg& arg) {
        return Query {std::make_unique<apq::query>((boost::format(Tag::text()) % arg).str())};
    }
};

using BeginQuery = Query<BeginQueryTag>;
using PerepareQuery = Query<PerepareQueryTag>;
using CommitPreparedQuery = Query<CommitPreparedQueryTag>;
using RollbackPreparedQuery = Query<RollbackPreparedQueryTag>;
using RollbackQuery = Query<RollbackQueryTag>;
using RegisterUserMaildbQuery = Query<RegisterUserMaildbQueryTag>;
using RegisterUserSharddbQuery = Query<RegisterUserSharddbQueryTag>;

template <typename RegisterUserQuery, typename ConnectionPoolHolder = apq::connection_pool_holder, typename RowIterator = apq::row_iterator>
class Participant: public std::enable_shared_from_this<Participant<RegisterUserQuery, ConnectionPoolHolder, RowIterator>> {
private:
    using ApqCallback = std::function<void(apq::result, RowIterator)>;

public:
    Participant(ConfigPtr config,
            const Scribe& scribe,
            const RegParams& params,
            const std::string& dbname,
            ConnectionPoolSharedPtr connectionPool,
            std::chrono::milliseconds timeout,
            const std::string& host,
            RegisterUserQuery registerQuery,
            OnceHandlerPtr onceHandler)
        : config_(config)
        , scribe_(scribe)
        , params_(params)
        , dbname_(dbname)
        , connectionPool_(connectionPool)
        , connection_(connectionPool_->pool())
        , timeout_(timeout)
        , host_(host)
        , registerQuery_(std::move(registerQuery))
        , onceHandler_(onceHandler)
        , state_(ParticipantState::Init)
        , commitTries_(0)
        , rollbackTries_(0)
        , transactionId_(transactionId())
        , logger_(getLogger(scribe_.logger, host_, dbname_, transactionId_))
    {}

    template <class ApqHandler>
    void start(ApqHandler&& handler) {
        request(BeginQuery {}, std::forward<ApqHandler>(handler));
    }

    template <class ApqHandler>
    void doRegister(ApqHandler&& handler) {
        request(std::move(registerQuery_), std::forward<ApqHandler>(handler));
    }

    template <class ApqHandler>
    void prepare(ApqHandler&& handler) {
        request(getPrepareQuery(), std::forward<ApqHandler>(handler));
    }

    template <class ApqHandler>
    void commit(ApqHandler&& handler) {
        request(getCommitPreparedQuery(), std::forward<ApqHandler>(handler));
    }

    template <class ApqHandler>
    void rollback(ApqHandler&& handler) {
        request(getRollbackPreparedQuery(), std::forward<ApqHandler>(handler));
    }

    template <class SyncPrepare, class ApqHandler>
    void onBegin(apq::result res, SyncPrepare&& syncPrepare, ApqHandler&& handler) {
        if (res.code()) {
            state_ = ParticipantState::PrepareError;
            LOGDOG_(logger_, error, log::where_name="begin", log::message=res.message());
            rollbackMyTransaction();
            syncPrepare();
        } else {
            doRegister(std::forward<ApqHandler>(handler));
        }
    }

    template <class SyncPrepare, class ApqHandler>
    void onRegister(apq::result res, RowIterator it, SyncPrepare&& syncPrepare,
            ApqHandler&& onPrepareHandler) {
        using namespace log;
        if (res.code()) {
            state_ = ParticipantState::PrepareError;
            LOGDOG_(logger_, error, where_name="register", message=res.message(), apq_result=res);
            rollbackMyTransaction();
            syncPrepare();
        } else if (it == RowIterator()) {
            state_ = ParticipantState::PrepareError;
            LOGDOG_(logger_, error, where_name="register", message="no rows in result", apq_result=res);
            rollbackMyTransaction();
            syncPrepare();
        } else if (it->size() < 1) {
            state_ = ParticipantState::PrepareError;
            LOGDOG_(logger_, error, where_name="register", message="result row size < 1", apq_result=res);
            rollbackMyTransaction();
            syncPrepare();
        } else {
            std::string registerResult;
            it->at(0, registerResult);
            if (registerResult == "already_registered") {
                std::string msg = fmt::format("user {} already registered in {}", params_.uid.value(), dbname_);
                callOnce(logger_, onceHandler_, msg, RegistrationError::userAlreadyRegistered, params_.uid.value());
                state_ = ParticipantState::PrepareError;
                rollbackMyTransaction();
                syncPrepare();
            } else if (registerResult == "already_in_progress") {
                std::string msg = fmt::format("registration for user {} is already in progress in {}",
                    params_.uid.value(), dbname_);
                callOnce(logger_, onceHandler_, msg, RegistrationError::registrationInProgress, params_.uid.value());
                state_ = ParticipantState::PrepareError;
                rollbackMyTransaction();
                syncPrepare();
            } else if (registerResult == "shard_is_occupied_by_user") {
                std::string msg = fmt::format("{} shard is already occupied by user {}, but it is not here", dbname_, params_.uid.value());
                callOnce(logger_, onceHandler_, msg, RegistrationError::shardIsOccupiedByUser, params_.uid.value());
                state_ = ParticipantState::PrepareError;
                rollbackMyTransaction();
                syncPrepare();
            } else if (registerResult == "success") {
                prepare(std::forward<ApqHandler>(onPrepareHandler));
            } else {
                state_ = ParticipantState::PrepareError;
                LOGDOG_(logger_, error, where_name="register",
                    message="unknown return result from register_user: " + registerResult);
                rollbackMyTransaction();
                syncPrepare();
            }
        }
    }

    template <class SyncPrepare>
    void onPrepare(apq::result res, SyncPrepare&& syncPrepare) {
        if (res.code()) {
            state_ = ParticipantState::PrepareError;
            LOGDOG_(logger_, error, log::where_name="prepare", log::message=res.message());
            rollbackMyPreparedTransaction();
        } else {
            state_ = ParticipantState::PrepareSuccess;
        }
        syncPrepare();
    }

    template <class ApqHandler>
    void onRollback(apq::result res, ApqHandler&& handler) {
        using namespace log;
        if (res.code()) {
            std::string msg = fmt::format("{}; tries {} from {}", res.message(), rollbackTries_, config_->registration.maxRollbackTries);
            LOGDOG_(logger_, error, where_name="rollback", message=msg);

            ++rollbackTries_;
            if (rollbackTries_ < config_->registration.maxRollbackTries) {
                rollback(std::forward<ApqHandler>(handler));
            } else {
                LOGDOG_(logger_, error, where_name="rollback",
                        message="no more tries for rollback; it now has stalled transaction");
            }
        } else {
            LOGDOG_(logger_, notice, where_name="rollback", message="successful rollback of prepared transaction");
        }
    }

    template <class SyncCommit, class ApqHandler>
    void onCommit(apq::result res, SyncCommit&& syncCommit, ApqHandler&& handler) {
        if (res.code()) {
            using namespace log;
            std::string msg = fmt::format("{}; tries {} from {}", res.message(), commitTries_, config_->registration.maxCommitTries);
            LOGDOG_(logger_, error, where_name="commit", message=msg);

            ++commitTries_;
            if (commitTries_ < config_->registration.maxCommitTries) {
                commit(std::forward<ApqHandler>(handler));
            } else {
                state_ = ParticipantState::CommitError;
                LOGDOG_(logger_, error, where_name="commit",
                    message="no more tries for commit; it now has stalled transaction");
            }
        } else {
            state_ = ParticipantState::CommitSuccess;
        }
        syncCommit();
    }

    ParticipantState state() const {
        return state_;
    }

    void state(ParticipantState val) {
        state_ = val;
    }

private:
    static auto getLogger(const Scribe::Logger& logger, const std::string& host, const std::string& dbname,
            const std::string& transactionId) {
        return logdog::bind(logger, log::where_file="register_executor", log::host=host, log::dbname=dbname,
            log::transaction_id=transactionId);
    }

    template <typename F>
    static void runAfterWait(unsigned attempt, unsigned delayBase, F f, boost::asio::io_service& ios) {
        if (0 == delayBase) {
            f();
            return;
        }

        const auto timer = std::make_shared<boost::asio::steady_timer>(ios);
        const int64_t delay = static_cast<int64_t>(std::pow(delayBase, attempt));
        timer->expires_after(std::chrono::seconds(delay));
        timer->async_wait([timer, f] (const boost::system::error_code& ec) {
            if (!ec) {
                f();
            }
        });
    }

    template <typename QueryTag>
    void rollbackHandler(apq::result res, QueryTag tag) {
        using namespace log;
        if (res.code()) {
            const std::string msg = fmt::format("{}; tries {} from {}",
                res.message(), rollbackTries_, config_->registration.maxRollbackTries);
            LOGDOG_(logger_, error, where_name=tag.name(), message=msg, apq_result=res);

            if constexpr (std::is_same_v<QueryTag, RollbackPreparedQueryTag>) {
                if (res.sqlstate_code() == apq::error::sqlstate::active_sql_transaction) {
                    return rollbackMyTransaction();
                }
            }

            if (++rollbackTries_ < config_->registration.maxRollbackTries) {
                runAfterWait(rollbackTries_, config_->registration.retryDelayBase, [self = this->shared_from_this()] () {
                    if constexpr (std::is_same_v<QueryTag, RollbackPreparedQueryTag>) {
                        self->rollbackMyPreparedTransaction();
                    } else if constexpr (std::is_same_v<QueryTag, RollbackQueryTag>) {
                        self->rollbackMyTransaction();
                    }
                }, getIoService());
            } else {
                LOGDOG_(logger_, error, where_name=tag.name(), message="no more tries left for rollback");
            }
        } else {
            LOGDOG_(logger_, notice, where_name=tag.name(), message="successful");
        }
    }

    void rollbackMyTransaction() {
        auto self = this->shared_from_this();

        request(RollbackQuery{}, [self](apq::result res, RowIterator) {
            self->rollbackHandler(std::move(res), RollbackQueryTag());
        });
    }

    void rollbackMyPreparedTransaction() {
        auto self = this->shared_from_this();

        rollback([self](apq::result res, RowIterator) {
            self->rollbackHandler(std::move(res), RollbackPreparedQueryTag());
        });
    }

    template <class Tag, class ApqHandler>
    void request(Query<Tag> wrap, ApqHandler&& handler) {
        LOGDOG_(logger_, notice, log::where_name="request", log::message=wrap.query->text_);
        query_ = std::move(wrap.query);
        Profile profile(scribe_.profiler, Tag::name(), host_);
        const ApqCallback profiled = asyncProfile(std::move(profile), std::forward<ApqHandler>(handler));
        connection_.async_request(apq::fake_task_context(), *query_, profiled, apq::result_format_binary, timeout_);
    }

    boost::asio::io_service& getIoService() {
        return connectionPool_->pool().get_io_service();
    }

    std::string transactionId() const {
        return fmt::format("reg_{}_u{}_s{}_{}", dbname_, params_.uid.value(), params_.shardId, params_.sessionId);
    }

    PerepareQuery getPrepareQuery() const {
        return PerepareQuery::format(transactionId_);
    }

    CommitPreparedQuery getCommitPreparedQuery() const {
        return CommitPreparedQuery::format(transactionId_);
    }

    RollbackPreparedQuery getRollbackPreparedQuery() const {
        return RollbackPreparedQuery::format(transactionId_);
    }

    using Logger = decltype(getLogger(
        std::declval<const Scribe::Logger&>(),
        std::declval<const std::string&>(),
        std::declval<const std::string&>(),
        std::declval<const std::string&>()));

    const ConfigPtr config_;
    const Scribe scribe_;
    const RegParams params_;
    const std::string dbname_;
    ConnectionPoolSharedPtr connectionPool_;
    ConnectionPoolHolder connection_;
    std::chrono::milliseconds timeout_;
    const std::string host_;
    RegisterUserQuery registerQuery_;
    const OnceHandlerPtr onceHandler_;
    std::atomic<ParticipantState> state_;
    unsigned commitTries_;
    unsigned rollbackTries_;
    std::string transactionId_;
    std::unique_ptr<apq::query> query_;
    Logger logger_;
};

template <typename ConnectionPoolHolder = apq::connection_pool_holder, typename RowIterator = apq::row_iterator>
class RegisterUserExecutor: public std::enable_shared_from_this<RegisterUserExecutor<ConnectionPoolHolder, RowIterator>>
                          , public EnableStaticConstructor<RegisterUserExecutor<ConnectionPoolHolder, RowIterator>> {
private:
    typedef RegisterUserExecutor<ConnectionPoolHolder, RowIterator> Self;
    typedef Participant<RegisterUserMaildbQuery, ConnectionPoolHolder, RowIterator> MaildbParticipant;
    typedef Participant<RegisterUserSharddbQuery, ConnectionPoolHolder, RowIterator> SharddbParticipant;
    typedef std::shared_ptr<MaildbParticipant> MaildbParticipantPtr;
    typedef std::shared_ptr<SharddbParticipant> SharddbParticipantPtr;
    friend class EnableStaticConstructor<Self>;
public:
    void execute() {
        mdbParticipant_->state(ParticipantState::PrepareInProgress);
        sharddbParticipant_->state(ParticipantState::PrepareInProgress);
        mdbParticipant_->start([self = thiz()] (auto res, auto) { self->onMdbBegin(std::move(res)); });
        sharddbParticipant_->start([self = thiz()] (auto res, auto) { self->onSharddbBegin(std::move(res)); });
    }
private:
    RegisterUserExecutor(ConfigPtr config,
            const Scribe& scribe,
            ShardPoolPtr shardPool,
            PeersPoolPtr peersPool,
            const RegParams& params,
            const Handler& handler)
        : config_(config)
        , scribe_(scribe)
        , shardPool_(shardPool)
        , peersPool_(peersPool)
        , params_(params)
        , handler_(handler)
        , onceHandler_(std::make_shared<OnceHandler>(detail::wrapHandler(handler)))
        , mdbParticipant_(std::make_shared<MaildbParticipant>(config, scribe, params, "mdb", mdbConnection(),
                config_->registration.mdbTimeout, params_.mdbMaster.host, getMdbRegisterQuery(),
                onceHandler_))
        , sharddbParticipant_(std::make_shared<SharddbParticipant>(config, scribe, params, "sharddb", sharddbConnection(),
                config_->registration.sharddbTimeout, params_.sharddbMasterHost, getSharddbRegisterQuery(),
                onceHandler_))
        , logger_(getLogger(scribe_.logger))
    {}

    void onMdbBegin(apq::result res) {
        mdbParticipant_->onBegin(res, [this] () { syncPrepare(); },
            [self = thiz()] (auto res, auto it) { self->onMdbRegister(std::move(res), std::move(it)); });
    }

    void onMdbRegister(apq::result res, RowIterator it) {
        mdbParticipant_->onRegister(res, it, [this] () { syncPrepare(); },
            [self = thiz()] (auto res, auto) { self->onMdbPrepare(std::move(res)); });
    }

    void onMdbPrepare(apq::result res) {
        mdbParticipant_->onPrepare(res, [this] () { syncPrepare(); });
    }

    void onSharddbBegin(apq::result res) {
        sharddbParticipant_->onBegin(res, [this] () { syncPrepare(); },
            [self = thiz()] (auto res, auto it) { self->onSharddbRegister(std::move(res), std::move(it)); });
    }

    void onSharddbRegister(apq::result res, RowIterator it) {
        sharddbParticipant_->onRegister(res, it, [this] () { syncPrepare(); },
            [self = thiz()] (auto res, auto) { self->onSharddbPrepare(std::move(res)); });
    }

    void onSharddbPrepare(apq::result res) {
        sharddbParticipant_->onPrepare(res, [this] () { syncPrepare(); });
    }

    void syncPrepare() {
        const ParticipantState mdbState = mdbParticipant_->state();
        const ParticipantState sharddbState = sharddbParticipant_->state();

        if (mdbState == ParticipantState::PrepareError && sharddbState == ParticipantState::PrepareError)
        {
            callOnceError(logger_, onceHandler_, "both mdb and sharddb prepare failed", RegistrationError::mdbAndSharddbPrepareFailed);
        }
        else if (mdbState == ParticipantState::PrepareError && sharddbState == ParticipantState::PrepareInProgress)
        {
            callOnceError(logger_, onceHandler_, "mdb prepare failed", RegistrationError::maildbRegistrationError);
        }
        else if (mdbState == ParticipantState::PrepareError && sharddbState == ParticipantState::PrepareSuccess)
        {
            callOnceError(logger_, onceHandler_, "mdb prepare failed", RegistrationError::maildbRegistrationError);
            sharddbRollbackOnce();
        }
        else if (mdbState == ParticipantState::PrepareInProgress && sharddbState == ParticipantState::PrepareError)
        {
            callOnceError(logger_, onceHandler_, "sharddb prepare failed", RegistrationError::sharddbRegistrationError);
        }
        else if (mdbState == ParticipantState::PrepareInProgress && sharddbState == ParticipantState::PrepareInProgress)
        {
            // wait
        }
        else if (mdbState == ParticipantState::PrepareInProgress && sharddbState == ParticipantState::PrepareSuccess)
        {
            // wait
        }
        else if (mdbState == ParticipantState::PrepareSuccess && sharddbState == ParticipantState::PrepareError)
        {
            callOnceError(logger_, onceHandler_, "sharddb prepare failed", RegistrationError::sharddbRegistrationError);
            mdbRollbackOnce();
        }
        else if (mdbState == ParticipantState::PrepareSuccess && sharddbState == ParticipantState::PrepareInProgress)
        {
            // wait
        }
        else if (mdbState == ParticipantState::PrepareSuccess && sharddbState == ParticipantState::PrepareSuccess)
        {
            startMdbCommitsOnce();
        }
        else
        {
            std::string msg = fmt::format("logic error: cannot be in this state while prepare phase: mdb {}, sharddb {}",
                unsigned(mdbState), unsigned(sharddbState));
            callOnceError(logger_, onceHandler_, msg, RegistrationError::illegalParticipantsState);
            LOGDOG_(logger_, error, log::message=msg);
        }
    }

    void mdbRollbackOnce() {
        if (mdbRollbackOnceBarrier_.testAndSet()) {
            mdbParticipant_->rollback([self = thiz()] (auto res, auto) { self->onMdbRollback(std::move(res)); });
        }
    }

    void onMdbRollback(apq::result res) {
        mdbParticipant_->onRollback(res, [self = thiz()] (auto res, auto) { self->onMdbRollback(std::move(res)); });
    }

    void sharddbRollbackOnce() {
        if (sharddbRollbackOnceBarrier_.testAndSet()) {
            sharddbParticipant_->rollback([self = thiz()] (auto res, auto) { self->onSharddbRollback(std::move(res)); });
        }
    }

    void onSharddbRollback(apq::result res) {
        sharddbParticipant_->onRollback(res, [self = thiz()] (auto res, auto) { self->onSharddbRollback(std::move(res)); });
    }

    void startMdbCommitsOnce() {
        if (startMdbCommitsOnceBarrier_.testAndSet()) {
            mdbParticipant_->state(ParticipantState::CommitInProgress);
            mdbParticipant_->commit([self = thiz()] (auto res, auto) { self->onMdbCommit(std::move(res)); });
        }
    }

    void onMdbCommit(apq::result res) {
        mdbParticipant_->onCommit(res, [this] () {syncCommit();}, [self = thiz()] (auto res, auto) { self->onMdbCommit(std::move(res)); });
    }

    void startSharddbCommitsOnce() {
        if (startSharddbCommitsOnceBarrier_.testAndSet()) {
            sharddbParticipant_->state(ParticipantState::CommitInProgress);
            sharddbParticipant_->commit([self = thiz()] (auto res, auto) { self->onSharddbCommit(std::move(res)); });
        }
    }

    void onSharddbCommit(apq::result res) {
        sharddbParticipant_->onCommit(res, [this] () { syncCommit(); }, [self = thiz()] (auto res, auto) { self->onSharddbCommit(std::move(res)); });
    }

    void syncCommit() {
        const ParticipantState mdbState = mdbParticipant_->state();
        const ParticipantState sharddbState = sharddbParticipant_->state();

        const auto onUnreachableState = [this, mdbState, sharddbState] () {
            std::string msg = fmt::format("logic error: cannot be in this state while commit phase: mdb {}, sharddb {}", unsigned(mdbState), unsigned(sharddbState));
            callOnceError(logger_, onceHandler_, msg, RegistrationError::illegalParticipantsState);
            LOGDOG_(logger_, error, log::message=msg);
        };


        if (mdbState == ParticipantState::CommitError && sharddbState == ParticipantState::CommitError)
        {
            onUnreachableState();
        }
        else if (mdbState == ParticipantState::CommitError && sharddbState == ParticipantState::CommitInProgress)
        {
            onUnreachableState();
        }
        else if (mdbState == ParticipantState::CommitError && sharddbState == ParticipantState::CommitSuccess)
        {
            onUnreachableState();
        }
        else if (mdbState == ParticipantState::CommitInProgress && sharddbState == ParticipantState::CommitError)
        {
            onUnreachableState();
        }
        else if (mdbState == ParticipantState::CommitInProgress && sharddbState == ParticipantState::CommitInProgress)
        {
            onUnreachableState();
        }
        else if (mdbState == ParticipantState::CommitInProgress && sharddbState == ParticipantState::CommitSuccess)
        {
            onUnreachableState();
        }
        else if (mdbState == ParticipantState::CommitError && sharddbState == ParticipantState::PrepareSuccess)
        {
            callOnceError(logger_, onceHandler_, "mdb commit failed", RegistrationError::maildbRegistrationError);
            sharddbRollbackOnce();
            mdbRollbackOnce();
        }
        else if (mdbState == ParticipantState::CommitSuccess && sharddbState == ParticipantState::PrepareSuccess)
        {
            startSharddbCommitsOnce();
        }
        else if (mdbState == ParticipantState::CommitInProgress && sharddbState == ParticipantState::PrepareSuccess)
        {
            // wait
        }
        else if (mdbState == ParticipantState::CommitSuccess && sharddbState == ParticipantState::CommitError)
        {
            callOnceError(logger_, onceHandler_, "sharddb commit failed", RegistrationError::sharddbRegistrationError);
            // leave garbage in mdb expressly, because it shouldn't harm
        }
        else if (mdbState == ParticipantState::CommitSuccess && sharddbState == ParticipantState::CommitInProgress)
        {
            // wait
        }
        else if (mdbState == ParticipantState::CommitSuccess && sharddbState == ParticipantState::CommitSuccess)
        {
            callOnce(logger_, onceHandler_, "", RegistrationError::ok, params_.uid.value());
        }
        else
        {
            onUnreachableState();
        }
    }

    ConnectionPoolSharedPtr mdbConnection() {
        using namespace sharpei::db;
        const Shard::Database::Address& mdb = params_.mdbMaster;
        const auto mdbConnInfo = ConnectionInfo(mdb.host, mdb.port, mdb.dbname, config_->base()->shard.adaptor.authInfo);
        return shardPool_->get(mdbConnInfo);
    }

    ConnectionPoolSharedPtr sharddbConnection() {
        using sharpei::db::ConnectionInfo;
        const auto sharddbConnInfo = ConnectionInfo(config_->base()->peers.adaptor.connInfoWOHost, params_.sharddbMasterHost);
        return peersPool_->get(sharddbConnInfo);
    }

    RegisterUserMaildbQuery getMdbRegisterQuery() const {
        using sharpei::db::bind;
        RegisterUserMaildbQuery wrap;
        bind(*wrap.query, params_.uid);
        wrap.query->bind_const_string(params_.country);
        wrap.query->bind_const_string(params_.lang);
        wrap.query->bind_const_int64(params_.enableWelcome && params_.needsWelcome);
        wrap.query->bind_const_string(params_.requestId);
        wrap.query->bind_const_string(params_.verstkaSessionKey);
        wrap.query->bind_const_int64(params_.baseFilters);
        return wrap;
    }

    RegisterUserSharddbQuery getSharddbRegisterQuery() const {
        using sharpei::db::bind;
        RegisterUserSharddbQuery wrap;
        bind(*wrap.query, params_.uid);
        wrap.query->bind_const_int64(params_.shardId);
        return wrap;
    }

    std::shared_ptr<Self> thiz() {
        return std::enable_shared_from_this<Self>::shared_from_this();
    }

    static auto getLogger(const Scribe::Logger& logger) {
        return logdog::bind(logger, log::where_file="register_executor");
    }

    using Logger = decltype(getLogger(std::declval<const Scribe::Logger&>()));

    const ConfigPtr config_;
    const Scribe scribe_;
    ShardPoolPtr shardPool_;
    PeersPoolPtr peersPool_;
    const RegParams params_;
    const Handler handler_;

    OnceHandlerPtr onceHandler_;

    MaildbParticipantPtr mdbParticipant_;
    SharddbParticipantPtr sharddbParticipant_;

    OnceBarrier mdbRollbackOnceBarrier_;
    OnceBarrier sharddbRollbackOnceBarrier_;
    OnceBarrier startMdbCommitsOnceBarrier_;
    OnceBarrier startSharddbCommitsOnceBarrier_;
    Logger logger_;
};

} // namespace db
} // namespace mail
} // namespace sharpei
