#pragma once

#include <internal/logger.h>
#include <internal/mail/config.h>
#include <internal/cache/cache.h>
#include <internal/db/adaptors/meta_adaptor.h>
#include <internal/mail/db/adaptors/reg_adaptor.h>
#include <internal/mail/user_journal/journal.h>
#include <internal/mail/user_journal/params.h>

#include <ymod_httpclient/call.h>

#include <yplatform/task_context.h>

#include <yandex/blackbox/blackbox2.h>

#include <boost/any.hpp>
#include <boost/asio/yield.hpp>

namespace sharpei {
namespace mail {
namespace registration {

using WeightedShardIds = sharpei::db::WeightedShardIds;
using DomainId = unsigned;

struct Params {
    BasicUserId<std::int64_t> uid;
    std::string userIp;
    std::string requestId;
    std::string verstkaSessionKey;
    std::string sessionId;
    bool enableWelcomeLetters;
    bool createBaseFilters;
};

class Performer : public std::enable_shared_from_this<Performer> {
public:
    using UserId = BasicUserId<std::int64_t>;
    using MetaAdaptorPtr = sharpei::db::MetaAdaptorPtr<UserId::Value>;
    using RegAdaptorPtr = mail::db::RegAdaptorPtr;
    using RegParams = mail::db::RegParams;
    using ShardsMasters = std::vector<std::pair<Shard::Id, Shard::Database::Address>>;

    Performer(std::shared_ptr<ymod_httpclient::cluster_call> httpClient,
        MetaAdaptorPtr metaAdaptor,
        cache::CachePtr cache,
        RegAdaptorPtr regAdaptor,
        ConfigPtr config,
        UserJournalPtr journal);

    template <class Handler>
    void perform(Params params, Handler&& handler) const;

    template <class Handler>
    void getMaster(Handler handler) const;

    template <class Handler>
    void getUserInfo(UserId uid, const std::string& userIp, const std::string& sessionId, Handler&& handler) const;

    template <class Handler>
    void getRegData(UserId uid, Handler handler) const;

    template <class Handler>
    void registerUser(const RegParams& params, Handler&& handler) const;

    template <class Handler>
    void getDeletedUserData(UserId uid, Handler handler) const;

    ShardsMasters getRandomShardsIdsWithAliveMaster(WeightedShardIds& weightedShardIds) const;

    ShardsMasters getAliveShardMaster(Shard::Id shardId) const;

    void logRegistration(const ::user_journal::parameters::id::RegParams& params) const;

private:
    std::shared_ptr<ymod_httpclient::cluster_call> httpClient_;
    MetaAdaptorPtr metaAdaptor_;
    cache::CachePtr cache_;
    RegAdaptorPtr regAdaptor_;
    ConfigPtr config_;
    UserJournalPtr journal_;
};

using PerformerPtr = std::shared_ptr<const Performer>;

namespace detail {

struct UserInfo {
    std::string country;
    std::string lang;
    DomainId domainId = 0;
    bool hasMailishAlias = false;
};

std::pair<ExplainedError, UserInfo> parseUserInfo(const yhttp::response& userInfo);
bool isPddUser(const UserInfo& userInfo);

template <class Handler>
struct Context {
    PerformerPtr performer;
    Params params;
    Handler handler;
    UserInfo userInfo;
    sharpei::db::RegData regData;
    std::string masterHost;
    Performer::ShardsMasters shards;
    Performer::ShardsMasters::const_iterator shardsIt;

    Context(PerformerPtr performer, const Params& params, Handler&& handler)
        : performer(std::move(performer)),
          params(params),
          handler(std::forward<Handler>(handler)) {
    }
};

template <class Handler>
class Continuation : public boost::asio::coroutine {
    using RegParams = mail::db::RegParams;
    using RegData = sharpei::db::RegData;

public:
    using ContextPtr = std::shared_ptr<Context<Handler>>;

    Continuation(ContextPtr context)
        : context(std::move(context)) {}

    void operator() (ExplainedError ec, boost::any args) {
        if (ec && ec != RegistrationError::shardIsOccupiedByUser) {
            context->handler(std::move(ec), Shard::Id());
            return;
        }
        try {
            reenter(this) {
                yield getUserInfo();
                std::tie(ec, context->userInfo) = parseUserInfo(boost::any_cast<yhttp::response&>(args));
                if (ec) {
                    context->handler(std::move(ec), Shard::Id());
                    return;
                }
                yield getMaster();
                context->masterHost = boost::any_cast<std::string&>(args);
                yield getRegData();
                if (boost::any_cast<RegData&>(args).userShardId) {
                    context->handler(ExplainedError(Error::ok), boost::any_cast<RegData&>(args).userShardId.get());
                } else {
                    context->regData = std::move(boost::any_cast<RegData&>(args));

                    yield getDeletedUserData();
                    if (auto& exShardId = boost::any_cast<boost::optional<Shard::Id>&>(args)) {
                        context->shards = context->performer->getAliveShardMaster(exShardId.get());
                    } else {
                        context->shards = context->performer->getRandomShardsIdsWithAliveMaster(context->regData.weightedShardIds);
                    }
                    context->shardsIt = context->shards.begin();
                    do {
                        if (context->shardsIt == context->shards.end()) {
                            context->handler(ExplainedError(Error::noShardWithAliveMaster), Shard::Id());
                            return;
                        }
                        yield registerUser();
                        ++context->shardsIt;
                    } while (ec == RegistrationError::shardIsOccupiedByUser);
                    context->handler(ExplainedError(Error::ok), boost::any_cast<const Shard::Id>(args));
                }
            }
        } catch (const boost::exception& exception) {
            std::string error = fmt::format("{} in {}",boost::diagnostic_information(exception), __PRETTY_FUNCTION__);
            context->handler(ExplainedError(Error::mailRegistrationException, error), Shard::Id());
        } catch (const std::exception& exception) {
            std::string error = fmt::format("{} in {}", exception.what(), __PRETTY_FUNCTION__);
            context->handler(ExplainedError(Error::mailRegistrationException, error), Shard::Id());
        }
    }

    RegParams makeRegParams(Shard::Id shardId, Shard::Database::Address shardMaster, const std::string& masterHost) const {
        return RegParams {
            context->params.uid,
            context->userInfo.country,
            context->userInfo.lang,
            context->params.enableWelcomeLetters,
            !isPddUser(context->userInfo) && !context->userInfo.hasMailishAlias,
            context->params.createBaseFilters,
            shardId,
            std::move(shardMaster),
            masterHost,
            context->params.requestId,
            context->params.verstkaSessionKey,
            context->params.sessionId
        };
    }

private:
    ContextPtr context;

    void getMaster() {
        context->performer->getMaster([self = *this](auto ec, auto response) mutable {
            self(ExplainedError(std::move(ec)), std::move(response));
        });
    }

    void getUserInfo() {
        context->performer->getUserInfo(context->params.uid, context->params.userIp, context->params.sessionId,
            [self = *this] (boost::system::error_code ec, auto response) mutable {
                self(ExplainedError(std::move(ec)), std::move(response));
            });
    }

    void getRegData() {
        context->performer->getRegData(context->params.uid, [self = *this](auto ec, auto value) mutable {
            self(std::move(ec), std::move(value));
        });
    }

    void registerUser() {
        Shard::Id shardId;
        Shard::Database::Address shardMaster;
        std::tie(shardId, shardMaster) = std::move(*context->shardsIt);
        const auto params = makeRegParams(shardId, std::move(shardMaster), context->masterHost);
        registerUser(params);
    }

    void registerUser(const RegParams& params) {
        context->performer->registerUser(params,
            [self = *this, params] (auto ec) mutable {
                self(std::move(ec), params.shardId);
                if (!ec) {
                    self.context->performer->logRegistration(params);
                }
            });
    }

    void getDeletedUserData() {
        context->performer->getDeletedUserData(context->params.uid,
            [self = *this] (auto ec, auto value) mutable {
                self(std::move(ec), std::move(value));
            });
    }
};

class TaskContext : public yplatform::task_context {
public:
    using yplatform::task_context::task_context;

    const std::string& get_name() const override final {
        static const std::string name = "sharpei_mail_registration";
        return name;
    }
};

} // namespace detail

template <class HandlerT>
void Performer::perform(Params params, HandlerT&& handler) const {
    using Handler = std::decay_t<HandlerT>;
    using Continuation = detail::Continuation<Handler>;
    using Context = detail::Context<Handler>;
    auto context = std::make_shared<Context>(shared_from_this(), std::move(params), std::forward<HandlerT>(handler));
    Continuation(std::move(context))(ExplainedError(Error::ok), boost::any());
}

template <class Handler>
void Performer::getMaster(Handler handler) const {
    auto onResult = [handler](auto value) mutable {
        handler(ExplainedError(Error::ok), std::move(value));
    };
    auto onError = [handler](auto error) mutable { handler(std::move(error), std::string{}); };
    metaAdaptor_->getMaster(std::move(onResult), std::move(onError));
}

template <class Handler>
void Performer::getUserInfo(UserId uid,
                            const std::string& userIp,
                            const std::string& sessionId,
                            Handler&& handler) const {
    bb::DBFields dbFields;
    dbFields << "account_info.country.uid" << "userinfo.lang.uid" << "subscription.suid.2" << "hosts.db_id.2";
    bb::Options options;
    options << dbFields << bb::optGetAllAliases;
    const auto query = bb::InfoRequest(sharpei::to_string(uid), userIp, options);
    std::string url = fmt::format("?{}", query);
    auto request = yhttp::request::GET(url);
    httpClient_->async_run(boost::make_shared<detail::TaskContext>(sessionId), std::move(request),
                           config_->base()->blackbox.http.options, std::forward<Handler>(handler));
}

template <class Handler>
void Performer::getRegData(UserId uid, Handler handler) const {
    using sharpei::db::RegData;
    auto onResult = [handler] (auto value) mutable { handler(ExplainedError(Error::ok), std::move(value)); };
    auto onError = [handler] (auto error) mutable { handler(std::move(error), RegData {}); };
    metaAdaptor_->getUserRegData(uid, std::move(onResult), std::move(onError));
}

template <class Handler>
void Performer::registerUser(const RegParams& params, Handler&& handler) const {
    regAdaptor_->registerUser(params, std::forward<Handler>(handler));
}

template <class Handler>
void Performer::getDeletedUserData(UserId uid, Handler handler) const {
    auto onResult = [handler] (auto shardId, auto) mutable {
        handler(ExplainedError(Error::ok), boost::optional<Shard::Id>(shardId));
    };
    auto onError = [handler] (auto error) mutable {
        if (error == Error::uidNotFound) {
            handler(ExplainedError(Error::ok), boost::optional<Shard::Id>());
        } else {
            handler(std::move(error), boost::optional<Shard::Id>());
        }
    };
    metaAdaptor_->getDeletedUserData(uid, std::move(onResult), std::move(onError));
}

} // namespace registration
} // namespace mail
} // namespace sharpei

#include <boost/asio/unyield.hpp>
