#pragma once

#include <internal/server/handlers/get_user.h>
#include <internal/mail/registration.h>
#include <internal/mail/user_journal/journal.h>

#include <boost/asio/yield.hpp>

namespace sharpei {
namespace mail {
namespace server {
namespace handlers {

class GetUserErrorHandler {
public:
    GetUserErrorHandler(ConfigPtr config, db::PeersPoolPtr peersPool, ::user_journal::ServicePtr userJournalService)
        : config(config), peersPool(peersPool), userJournalService(userJournalService) {}

    template <class AsyncHandler>
    void operator ()(std::shared_ptr<AsyncHandler> asyncHandler, const ExplainedError& error) {
        using namespace ymod_webserver::helpers;
        using namespace ymod_webserver::helpers::transfer_encoding;
        reenter(asyncHandler->coroutine()) {
            while (true) {
                if (error == Error::uidNotFound) {
                    LOGDOG_(asyncHandler->req().scribe.logger, warning, log::error_code=error);
                    yield performRegistration(asyncHandler);
                }
                if (error == Error::blackBoxFatalError || error == Error::notPgUser) {
                    LOGDOG_(asyncHandler->req().scribe.logger, error, log::error_code=error);
                    const auto message = error.full_message();
                    Response(*asyncHandler->req().response).bad_request(fixed_size(format::json(message)));
                    return;
                } else if (error == Error::uidNotFound) {
                    LOGDOG_(asyncHandler->req().scribe.logger, error, log::error_code=error);
                    const auto message = error.full_message();
                    Response(*asyncHandler->req().response).not_found(fixed_size(format::json(message)));
                    return;
                } else if (retryNumber < maxRetriesNumber && error == RegistrationError::userAlreadyRegistered) {
                    ++retryNumber;
                    LOGDOG_(asyncHandler->req().scribe.logger, warning, log::error_code=error);
                    yield asyncHandler->execute();
                } else {
                    break;
                }
            }
            LOGDOG_(asyncHandler->req().scribe.logger, error, log::error_code=error);
            const auto message = error.full_message();
            Response(*asyncHandler->req().response).internal_server_error(fixed_size(format::json(message)));
        }
    }

private:
    static constexpr std::size_t maxRetriesNumber = 1;

    ConfigPtr config;
    db::PeersPoolPtr peersPool;
    std::size_t retryNumber = 0;
    ::user_journal::ServicePtr userJournalService;

    template <class AsyncHandler>
    void performRegistration(std::shared_ptr<AsyncHandler> asyncHandler) const {
        using db::getRegAdaptor;
        auto journal = getUserJournal(getUserJournalParams(asyncHandler->req()),
                                      "sharpei", userJournalService);
        const auto registration = std::make_shared<mail::registration::Performer>(
            yplatform::find<ymod_httpclient::cluster_call, std::shared_ptr>(config->base()->blackbox.http.module),
            asyncHandler->metaAdaptor(),
            asyncHandler->cache(),
            getRegAdaptor(config, asyncHandler->req().scribe, asyncHandler->shardPool(), peersPool),
            config, journal
        );
        mail::registration::Params params{
            asyncHandler->params().uid,
            asyncHandler->req().userIp(),
            asyncHandler->req().requestId(),
            asyncHandler->req().getHeader("X-Yandex-Session-Key"),
            asyncHandler->req().sessionId(),
            config->registration.enableWelcomeLetters,
            config->registration.createBaseFilters,
        };
        auto handler = [asyncHandler, config = config] (auto error, auto shardId) {
            if (error) {
                asyncHandler->onError(error);
            } else {
                asyncHandler->onGetUserData(shardId, boost::optional<std::string>());
            }
        };
        registration->perform(std::move(params), asyncHandler->wrap(std::move(handler)));
    }
};

class GetDeletedUserErrorHandler {
public:
    template <class AsyncHandler>
    void operator ()(std::shared_ptr<AsyncHandler> asyncHandler, const ExplainedError& error) {
        using namespace ymod_webserver::helpers;
        using namespace ymod_webserver::helpers::transfer_encoding;
        reenter(asyncHandler->coroutine()) {
            LOGDOG_(asyncHandler->req().scribe.logger, error, log::error_code=error);
            const auto message = error.full_message();
            if (error == Error::uidNotFound) {
                Response(*asyncHandler->req().response).not_found(fixed_size(format::json(message)));
            } else {
                Response(*asyncHandler->req().response).internal_server_error(fixed_size(format::json(message)));
            }
        }
    }
};

using GetUser = sharpei::server::handlers::BaseGetUser<
        GetUserErrorHandler,
        sharpei::server::handlers::ResponseShard,
        std::int64_t,
        sharpei::db::UserType::existing,
        sharpei::db::QueryType::withoutData>;

using GetDeletedUser = sharpei::server::handlers::BaseGetUser<
        GetDeletedUserErrorHandler,
        sharpei::server::handlers::ResponseShard,
        std::int64_t,
        sharpei::db::UserType::deleted,
        sharpei::db::QueryType::withoutData>;

} // namespace handlers
} // namespace server
} // namespace mail
} // namespace sharpei

#include <boost/asio/unyield.hpp>
