#pragma once

#include <internal/cache/cache.h>
#include <internal/db/adaptors/meta_adaptor.h>
#include <internal/db/adaptors/meta_adaptor_factory.h>
#include <internal/db/config.h>
#include <internal/db/ep/endpoint_provider.h>
#include <internal/db/pools/meta_pool.h>
#include <internal/poller/poller.h>
#include <internal/poller/shards_poller_config.h>
#include <internal/poller/shards_provider.h>

#include <memory>

namespace sharpei::poller {

class ShardsPoller {
public:
    ShardsPoller(const ShardsPollerConfig& config,
                 const db::MetaConfig& metaConfig,
                 const db::ShardConfig& shardConfig,
                 db::EndpointProviderPtr metaEndpointProvider,
                 db::ShardPoolPtr shardPool,
                 cache::CachePtr shardsCache,
                 db::MetaAdaptorFactoryPtr metaAdaptorFactory) {
        auto uniqId = "shards_poller";
        boost::uuids::random_generator generator;
        const Scribe scribe(uniqId, generateRequestId(generator));

        auto poolCfg = db::PoolConfig(metaConfig.db.pool);
        poolCfg.maxConnections = config.metaMaxConnections;
        auto pool = db::makeConnectionPool(poolCfg, io);
        const auto metaPool = std::make_shared<db::MetaPool>(std::move(pool), std::move(metaEndpointProvider));

        impl = std::make_shared<Poller>(
            poller::Poller::Type::shards,
            uniqId,
            config.base,
            makeShardsProvider(
                metaAdaptorFactory->getBaseMetaAdaptor(metaConfig.db.adaptor, metaPool, scribe)),
            db::getShardAdaptor(shardConfig.adaptor, shardPool, scribe, shardsCache),
            shardsCache);

        worker = std::make_unique<IoThreadWorker>(std::move(uniqId), io);
    }

    void start() {
        impl->start(io);
    }

    void stop() {
        impl->stop();
        worker->stop();
        io.stop();
    }

private:
    boost::asio::io_context io;
    std::unique_ptr<IoThreadWorker> worker;
    std::shared_ptr<Poller> impl;
    boost::uuids::random_generator generator;
};

} // namespace sharpei::poller
