#pragma once

#include <internal/random.h>
#include <internal/shard.h>

#include <util/system/compiler.h>
#include <yamail/data/reflection/reflection.h>

#include <boost/range/algorithm/transform.hpp>

namespace sharpei {
namespace reflection {

struct ShardWithoutRoles {
    using Addresses = std::vector<sharpei::Shard::Database::Address>;

    sharpei::Shard::Id id;
    sharpei::Shard::Name name;
    Addresses addrs;
};

struct Shard {
    using Databases = std::vector<sharpei::Shard::Database>;

    sharpei::Shard::Id id;
    sharpei::Shard::Name name;
    Databases databases;
};

class AddressWithStringPort {
public:
    AddressWithStringPort(sharpei::Shard::Database::Address impl)
        : impl_(std::move(impl)) {}

    const std::string& host() const {
        return impl_.host;
    }

    std::string port() const {
        return std::to_string(impl_.port);
    }

    const std::string& dbname() const {
        return impl_.dbname;
    }

    const std::string& dataCenter() const {
        return impl_.dataCenter;
    }

private:
    sharpei::Shard::Database::Address impl_;
};

class DatabaseWithStringPort {
public:
    using Role = sharpei::Shard::Database::Role;
    using Status = sharpei::Shard::Database::Status;
    using State = sharpei::Shard::Database::State;

    DatabaseWithStringPort(sharpei::Shard::Database impl)
        : impl_(std::move(impl)) {}

    AddressWithStringPort address() const {
        return AddressWithStringPort(impl_.address());
    }

    Role role() const {
        return impl_.role();
    }

    Status status() const {
        return impl_.status();
    }

    const State& state() const {
        return impl_.state();
    }

private:
    sharpei::Shard::Database impl_;
};

class ShardWithStringIdAndPort {
public:
    using Databases = std::vector<DatabaseWithStringPort>;

    ShardWithStringIdAndPort(Shard impl)
        : impl_(std::move(impl)) {}

    std::string id() const {
        return std::to_string(impl_.id);
    }

    const sharpei::Shard::Name& name() const {
        return impl_.name;
    }

    Databases databases() const {
        return Databases(impl_.databases.begin(), impl_.databases.end());
    }

private:
    Shard impl_;
};

namespace detail {

template <class Comparator = std::function<bool(const sharpei::Shard::Database&, const sharpei::Shard::Database&)>>
inline Comparator getComparator(Mode mode) {
    switch (mode.value) {
        case Mode::WriteRead:
            return [](const auto& lhs, const auto& rhs) {
                return std::pair(lhs.role(), lhs.state()) < std::pair(rhs.role(), rhs.state());
            };
        case Mode::ReadWrite:
            return [](const auto& lhs, const auto& rhs) {
                return std::pair(-lhs.role(), lhs.state()) < std::pair(-rhs.role(), rhs.state());
            };
        default:
            Y_UNREACHABLE();
    }
}

template <class Pred>
inline Shard::Databases makeDatabases(const sharpei::Shard::Databases& databases, Pred cmp) {
    Shard::Databases sequence(databases.begin(), databases.end());

    /* Client code usually just takes the first replica from the output to make a request.
       As the result the second replica is almost unused.
       To solve this it is sufficient to make the order inside equivalence classes
       (e.g. all the replicas with the same lag) random. */
    std::shuffle(sequence.begin(), sequence.end(), threadLocalGenerator());

    std::stable_sort(sequence.begin(), sequence.end(), cmp);
    return sequence;
}

inline Shard::Databases makeDatabasesOrderedByState(const sharpei::Shard::Databases& databases) {
    auto cmp = [](const auto& lhs, const auto& rhs) {
        return std::pair(lhs.state(), lhs.role()) < std::pair(rhs.state(), rhs.role());
    };
    return makeDatabases(databases, std::move(cmp));
}

}  // namespace detail

inline ShardWithoutRoles makeShardWithoutRolesOrderedByState(const sharpei::Shard& shard) {
    const auto sequence = detail::makeDatabasesOrderedByState(shard.databases);

    ShardWithoutRoles::Addresses addresses;
    for (const auto& db : sequence) {
        addresses.push_back(db.address());
    }
    return {shard.id, shard.name, std::move(addresses)};
}

inline Shard makeShardOrderedByState(const sharpei::Shard& shard) {
    return {shard.id, shard.name, detail::makeDatabasesOrderedByState(shard.databases)};
}

inline Shard makeShardOrderedByModeAndState(const sharpei::Shard& shard, Mode mode) {
    auto cmp = detail::getComparator(mode);
    return {shard.id, shard.name, detail::makeDatabases(shard.databases, std::move(cmp))};
}

template <class ShardT>
using ShardsInfoNewFormat = std::map<sharpei::Shard::Id, ShardT>;

typedef std::map<sharpei::Shard::Id, std::vector<DatabaseWithStringPort>> ShardsInfoOldFormat;

template <class ShardT>
ShardsInfoNewFormat<ShardT> makeShardsInfoNewFormat(const sharpei::ShardsInfoNewFormat& shardsInfo) {
    ShardsInfoNewFormat<ShardT> result;

    boost::transform(shardsInfo, std::inserter(result, result.end()),
        [&] (const auto& shard) -> typename ShardsInfoNewFormat<ShardT>::value_type {
            return {shard.first, makeShardOrderedByState(shard.second)};
        });

    return result;
}

inline ShardsInfoOldFormat makeShardsInfoOldFormat(const sharpei::ShardsInfoOldFormat& shardsInfo) {
    ShardsInfoOldFormat result;

    boost::transform(shardsInfo, std::inserter(result, result.end()),
        [&] (const auto& shard) -> ShardsInfoOldFormat::value_type {
            auto databases = detail::makeDatabasesOrderedByState(shard.second);
            return {shard.first, std::vector<DatabaseWithStringPort>(std::make_move_iterator(databases.begin()),
                                                                     std::make_move_iterator(databases.end()))};
        });

    return result;
}

} // namespace reflection
} // namespace sharpei

BOOST_FUSION_ADAPT_STRUCT(sharpei::Shard::Database::Address,
    (std::string, host)
    (unsigned, port)
    (std::string, dbname)
    (std::string, dataCenter)
)

BOOST_FUSION_ADAPT_STRUCT(sharpei::Shard::Database::State,
    (sharpei::Shard::Database::State::ReplicationLag, lag)
)

YREFLECTION_ADAPT_ADT(sharpei::Shard::Database,
    YREFLECTION_ROPROPERTY(sharpei::Shard::Database::Address, address)
    (role  , std::string, std::string, obj.role().toString()  , (void)obj; (void)val )
    (status, std::string, std::string, obj.status().toString(), (void)obj; (void)val )
    YREFLECTION_ROPROPERTY(sharpei::Shard::Database::State, state)
)

BOOST_FUSION_ADAPT_STRUCT(sharpei::reflection::ShardWithoutRoles,
    (sharpei::Shard::Id, id)
    (sharpei::Shard::Name, name)
    (sharpei::reflection::ShardWithoutRoles::Addresses, addrs)
)

BOOST_FUSION_ADAPT_STRUCT(sharpei::reflection::Shard,
    (sharpei::Shard::Id, id)
    (sharpei::Shard::Name, name)
    (sharpei::reflection::Shard::Databases, databases)
)

BOOST_FUSION_ADAPT_STRUCT(sharpei::Shard,
    (sharpei::Shard::Id, id)
    (sharpei::Shard::Name, name)
    (sharpei::Shard::Databases, databases)
)

YREFLECTION_ADAPT_ADT(sharpei::reflection::AddressWithStringPort,
    YREFLECTION_ROPROPERTY(std::string, host)
    YREFLECTION_ROPROPERTY(std::string, port)
    YREFLECTION_ROPROPERTY(std::string, dbname)
    YREFLECTION_ROPROPERTY(std::string, dataCenter)
)

YREFLECTION_ADAPT_ADT(sharpei::reflection::DatabaseWithStringPort,
    YREFLECTION_ROPROPERTY(sharpei::reflection::AddressWithStringPort, address)
    (role  , std::string, std::string, obj.role().toString()  , (void)obj; (void)val)
    (status, std::string, std::string, obj.status().toString(), (void)obj; (void)val)
    YREFLECTION_ROPROPERTY(sharpei::Shard::Database::State, state)
)

YREFLECTION_ADAPT_ADT(sharpei::reflection::ShardWithStringIdAndPort,
    YREFLECTION_ROPROPERTY(std::string, id)
    YREFLECTION_ROPROPERTY(sharpei::Shard::Name, name)
    YREFLECTION_ROPROPERTY(sharpei::reflection::ShardWithStringIdAndPort::Databases, databases)
)

BOOST_FUSION_ADAPT_STRUCT(sharpei::ShardWithoutRoles,
    id,
    name,
    addrs
)
