#ifndef SHARPEI_SERVER_HANDLERS_BASE_H
#define SHARPEI_SERVER_HANDLERS_BASE_H

#include <internal/server/request_context.h>
#include <internal/logger.h>
#include <internal/config.h>
#include <internal/cache/cache.h>
#include <ymod_webserver/codes.h>
#include <ymod_webserver_helpers/helpers.hpp>

namespace sharpei {
namespace server {
namespace handlers {

using namespace ymod_webserver::helpers;
using namespace ymod_webserver::helpers::transfer_encoding;

namespace detail {

inline void logError(
    RequestContext& request,
    const std::string& msg,
    const char* function,
    const char* file,
    int line) {
    using namespace log;
    LOGDOG_(request.scribe.logger, error, method=request.method, message=msg,
            where_file=file, where_name=function, where_line=std::to_string(line));
}

template <typename Action>
inline void safeExec(RequestContext& request, const char* function, const char* file, int line, Action action) {
    try {
        action();
    } catch (const ParamsException& e) {
        const std::string msg = std::string("failed to parse params: ") + e.what();
        Response(*request.response).bad_request(fixed_size(format::text(msg)));
        logError(request, msg, function, file, line);
    } catch (const std::exception& e) {
        const std::string msg = std::string("exception: ") + e.what();
        Response(*request.response).internal_server_error(fixed_size(format::text(msg)));
        logError(request, msg, function, file, line);
    } catch (...) {
        const std::string msg("unknown error");
        Response(*request.response).internal_server_error(fixed_size(format::text(msg)));
        logError(request, msg, function, file, line);
    }
}

} // namespace detail

#define SHARPEI_SAFE_EXEC(request, action) \
    sharpei::server::handlers::detail::safeExec(request, __PRETTY_FUNCTION__, __FILE__, __LINE__, action)

class Base {
public:
    virtual ~Base() {}

    ymod_webserver::methods::http_method getMethod() const {
        return method();
    }

    void process(const ymod_webserver::http::stream_ptr& stream) {
        const auto request = stream->request();
        if (request->method != method()) {
            stream->result(ymod_webserver::codes::method_not_allowed);
            return;
        }
        RequestContext context(request, stream, request->url.make_full_path());
        SHARPEI_SAFE_EXEC(context, ([&] { execute(context); }));
    }

protected:
    virtual ymod_webserver::methods::http_method method() const = 0;
    virtual void execute(RequestContext& req) const = 0;
};

} // namespace server
} // namespace handlers
} // namespace sharpei

#endif // SHARPEI_SERVER_HANDLERS_BASE_H
