#pragma once

#include <boost/any.hpp>
#include <boost/asio/coroutine.hpp>
#include <boost/optional.hpp>

#include <internal/config.h>
#include <internal/db/adaptors/meta_adaptor.h>
#include <internal/db/adaptors/peers_adaptor.h>
#include <internal/reflection/shard.h>
#include <internal/server/handlers/base.h>
#include <internal/server/request_context.h>
#include <internal/random_shard_id_with_alive_master.h>

#include <boost/asio/yield.hpp>

namespace sharpei {
namespace server {
namespace handlers {

template <class PeersAdaptorT, class UserIdValue>
class CreateUserPerformer : public std::enable_shared_from_this<CreateUserPerformer<PeersAdaptorT, UserIdValue>> {
public:
    using UserId = BasicUserId<UserIdValue>;
    using MetaAdaptorPtr = db::MetaAdaptorPtr<typename UserId::Value>;
    using PeersAdaptor = PeersAdaptorT;
    using RegData = db::RegData;

    CreateUserPerformer(
            RequestContext context,
            ConfigPtr config,
            cache::CachePtr cache,
            MetaAdaptorPtr metaAdaptor,
            PeersAdaptor peersAdaptor);

    void perform();

private:
    using OptShardMaster = boost::optional<std::pair<Shard::Id, Shard::Database::Address>>;
    using WeightedShardIds = db::WeightedShardIds;

    RequestContext context_;
    ConfigPtr config_;
    cache::CachePtr cache_;
    MetaAdaptorPtr metaAdaptor_;
    PeersAdaptor peersAdaptor_;
    UserId uid_;
    boost::optional<Shard::Id> shardId_;
    std::string masterHost_;
    boost::asio::coroutine coroutine_;

    void continuation(ExplainedError error, boost::any args);

    void getRegData();
    void getMaster();
    void createUser(RegData& regData);
    void createUser(const std::string& masterHost, const Shard::Id shardId);

    void finish(const Shard::Id& shard) const;
    void finish(const Shard& shard) const;
    void finish(const ExplainedError& error) const;

    std::optional<Shard::Id> getRandomShardIdWithAliveMaster(WeightedShardIds&& weightedShardIds) const;
};

template <class P, class U>
CreateUserPerformer<P, U>::CreateUserPerformer(RequestContext context,
    ConfigPtr config,
    cache::CachePtr cache,
    MetaAdaptorPtr metaAdaptor,
    PeersAdaptor peersAdaptor)
        : context_(std::move(context)),
          config_(std::move(config)),
          cache_(std::move(cache)),
          metaAdaptor_(std::move(metaAdaptor)),
          peersAdaptor_(std::move(peersAdaptor)) {
}

template <class P, class U>
void CreateUserPerformer<P, U>::perform()  {
    const auto& params = context_.request->url.params;
    const auto uid = params.find("uid");
    if (uid == params.end()) {
        return finish(ExplainedError(Error::invalidRequest, "uid parameter not found"));
    }
    bool ok;
    std::tie(ok, uid_) = lexicalCast<typename UserId::Value>(uid->second);
    if (!ok) {
        return finish(ExplainedError(Error::invalidRequest, "invalid uid parameter value"));
    }
    const auto shardId = params.find("shard_id");
    if (shardId != params.end()) {
        std::tie(ok, shardId_) = lexicalCast<Shard::Id>(shardId->second);
        if (!ok) {
            return finish(ExplainedError(Error::invalidRequest, "invalid shard_id parameter value"));
        }
    }
    continuation(ExplainedError(Error::ok), boost::any());
}

template <class P, class U>
void CreateUserPerformer<P, U>::continuation(ExplainedError error, boost::any args) {
    try {
        if (error) {
            return finish(std::move(error));
        }
        Shard::Id shardId = 0;
        reenter(coroutine_) {
            yield getMaster();
            masterHost_ = boost::any_cast<std::string&>(args);
            if (shardId_) {
                yield createUser(masterHost_, shardId_.get());
            } else {
                yield getRegData();
                if (const auto shardId = boost::any_cast<RegData&>(args).userShardId) {
                    LOGDOG_(context_.scribe.logger, notice, log::message="user exists in shard", log::shard_id=shardId.get());
                    return finish(shardId.get());
                }
                yield createUser(boost::any_cast<RegData&>(args));
            }
            shardId = boost::any_cast<const Shard::Id>(args);
            LOGDOG_(context_.scribe.logger, notice, log::message="user created in shard", log::shard_id=shardId);
            finish(shardId);
        }
    } catch (const boost::exception& exception) {
        finish(ExplainedError(Error::internalError, boost::diagnostic_information(exception)));
    } catch (const std::exception& exception) {
        finish(ExplainedError(Error::internalError, exception.what()));
    } catch (...) {
        finish(ExplainedError(Error::internalError, "unknown error"));
    }
}

template <class P, class U>
void CreateUserPerformer<P, U>::getRegData() {
    const auto self = this->shared_from_this();
    auto onResult = [self] (auto value) { self->continuation(ExplainedError(Error::ok), std::move(value)); };
    auto onError = [self] (auto error) { self->continuation(std::move(error), boost::any()); };
    metaAdaptor_->getUserRegData(uid_, std::move(onResult), std::move(onError));
}

template <class P, class U>
void CreateUserPerformer<P, U>::getMaster() {
    const auto self = this->shared_from_this();
    auto onResult = [self] (auto value) { self->continuation(ExplainedError(Error::ok), std::move(value)); };
    auto onError = [self] (auto error) { self->continuation(std::move(error), boost::any()); };
    metaAdaptor_->getMaster(std::move(onResult), std::move(onError));
}

template <class P, class U>
void CreateUserPerformer<P, U>::createUser(RegData& regData) {
    if (const auto shardId = getRandomShardIdWithAliveMaster(std::move(regData.weightedShardIds))) {
        createUser(masterHost_, *shardId);
    } else {
        finish(ExplainedError(Error::noShardWithAliveMaster));
    }
}

template <class P, class U>
void CreateUserPerformer<P, U>::createUser(const std::string& masterHost, Shard::Id shardId) {
    const db::CreateUserParams<typename UserId::Value> params {uid_, shardId};
    const auto self = this->shared_from_this();
    LOGDOG_(context_.scribe.logger, notice, log::message="create user in shard", log::shard_id=shardId);
    peersAdaptor_.createUser(masterHost, params,
        [self] (auto error, auto shardId) { self->continuation(std::move(error), shardId); });
}

template <class P, class U>
std::optional<Shard::Id> CreateUserPerformer<P, U>::getRandomShardIdWithAliveMaster(
        WeightedShardIds&& weightedShardIds) const {
    return sharpei::getRandomShardIdWithAliveMaster(*cache_, std::move(weightedShardIds));
}

template <class P, class U>
void CreateUserPerformer<P, U>::finish(const Shard::Id& shardId) const {
    ExplainedError error;
    boost::optional<Shard> shard;
    std::tie(error, shard) = cache_->getShard(shardId);
    if (error) {
        finish(error);
    } else {
        finish(shard.get());
    }
}

template <class P, class U>
void CreateUserPerformer<P, U>::finish(const Shard& shard) const {
    using reflection::makeShardOrderedByState;
    const auto ordered = makeShardOrderedByState(shard);
    Response(*context_.response).ok(fixed_size(format::json(ordered)));
}

template <class P, class U>
void CreateUserPerformer<P, U>::finish(const ExplainedError& error) const {
    using namespace ymod_webserver::helpers;
    using namespace ymod_webserver::helpers::transfer_encoding;
    if (error == Error::invalidRequest) {
        LOGDOG_(context_.scribe.logger, warning, log::error_code=error);
        Response(*context_.response).bad_request(fixed_size(format::json(error)));
    } else {
        LOGDOG_(context_.scribe.logger, error, log::error_code=error);
        Response(*context_.response).internal_server_error(fixed_size(format::json(error)));
    }
}

} // namespace handlers
} // namespace server
} // namespace sharpei

#include <boost/asio/unyield.hpp>
