#pragma once

#include <boost/asio/spawn.hpp>

#include <internal/reflection/shard.h>
#include <internal/server/request_context.h>
#include <internal/db/adaptors/meta_adaptor.h>
#include <internal/db/adaptors/peers_adaptor.h>
#include <internal/config.h>
#include <internal/expected.h>
#include <internal/random.h>
#include <internal/random_shard_id_with_alive_master.h>
#include <internal/services/blackbox/blackbox.h>

#include <ymod_webserver_helpers/helpers.hpp>

namespace sharpei::server::handlers {

using sharpei::services::blackbox::BlackboxPtr;
using sharpei::server::RequestContext;

template <class MetaAdaptor, class PeersAdaptor>
class GetDomainPerformer {
public:
    GetDomainPerformer(RequestContext context, ConfigPtr config, MetaAdaptor metaAdaptor,
            PeersAdaptor peersAdaptor, cache::CachePtr cache, BlackboxPtr blackbox)
        : context(std::move(context)),
          config(std::move(config)),
          metaAdaptor(std::move(metaAdaptor)),
          peersAdaptor(std::move(peersAdaptor)),
          cache(std::move(cache)),
          blackbox(std::move(blackbox))
    {}

    void operator ()(boost::asio::yield_context yield) const {
        getDomainIdFromContext()
            .bind([&] (const auto domainId) { return getDomainShardId(domainId, yield); })
            .bind([&] (const auto shardId) { return getShardFromCache(shardId); })
            .bind([&] (const auto& shard) { return finish(shard); })
            .catch_error([&] (const auto& error) { return finish(error); });
    }

private:
    template <class T>
    using IoResultYield = io_result::basic_yield_context<T>;

    RequestContext context;
    ConfigPtr config;
    MetaAdaptor metaAdaptor;
    PeersAdaptor peersAdaptor;
    cache::CachePtr cache;
    BlackboxPtr blackbox;

    expected<DomainId> getDomainIdFromContext() const {
        const auto& params = context.request->url.params;
        const auto domainId = params.find("domain_id");
        if (domainId == params.end()) {
            return make_unexpected(ExplainedError(Error::invalidRequest, "domain_id parameter not found"));
        }
        auto [ok, result] = lexicalCast<DomainId>(domainId->second);
        if (!ok) {
            return make_unexpected(ExplainedError(Error::invalidRequest, "invalid domain_id parameter value"));
        }
        return result;
    }

    expected<Shard::Id> getDomainShardId(const DomainId domainId, boost::asio::yield_context yield) const {
        ExplainedError ec;
        const auto result = metaAdaptor.getDomainShardId(domainId, io_result::make_yield_context(yield)[ec]);
        if (!ec) {
            return result;
        } else if (ec != Error::domainNotFound) {
            return make_unexpected(std::move(ec));
        } else {
            LOGDOG_(context.scribe.logger, notice, log::error_code=ec);
            return isDomainExistsInBlackbox(domainId, yield)
                .bind([&] (const auto domainExist) -> expected<Shard::Id> {
                    if (!domainExist) {
                        return make_unexpected(std::move(ec));
                    }
                    LOGDOG_(context.scribe.logger, notice, log::message="domain is found in blackbox");
                    return createDomain(domainId, io_result::make_yield_context(yield)[ec]);
                });
        }
    }

    expected<bool> isDomainExistsInBlackbox(const DomainId domainId, boost::asio::yield_context yield) const {
        const auto blackboxContext = makeTaskContext(context.sessionId(), context.requestId(), yield);
        return blackbox->getHostedDomains(domainId, blackboxContext)
            .bind([&] (const auto& hostedDomains) { return !hostedDomains.empty(); });
    }

    template <class Handler>
    expected<Shard::Id> createDomain(const DomainId domainId, IoResultYield<Handler> yield) const {
        return getMaster(yield).bind([&](auto masterHost) {
            return getRegData(yield).bind(
                [&](auto regData) { return createDomain(domainId, std::move(masterHost), std::move(regData), yield); });
        });
    }

    template <class Handler>
    expected<Shard::Id> createDomain(const DomainId domainId,
                                     std::string masterHost,
                                     db::RegData regData,
                                     IoResultYield<Handler> yield) const {
        if (const auto shardId = getRandomShardIdWithAliveMaster(std::move(regData.weightedShardIds))) {
            db::CreateDomainParams params;
            params.domainId = domainId;
            params.shardId = *shardId;
            ExplainedError ec;
            const auto realShardId = peersAdaptor.createDomain(masterHost, params, yield[ec]);
            if (ec) {
                return make_unexpected(std::move(ec));
            }
            if (*shardId == realShardId) {
                LOGDOG_(context.scribe.logger, notice, log::message="domain created", log::shard_id=realShardId);
            }
            return realShardId;
        }
        return make_unexpected(ExplainedError(Error::noShardWithAliveMaster));
    }

    template <class Handler>
    expected<std::string> getMaster(IoResultYield<Handler> yield) const {
        ExplainedError ec;
        auto masterHost = metaAdaptor.getMaster(yield[ec]);
        if (ec) {
            return make_unexpected(std::move(ec));
        }
        return masterHost;
    }

    template <class Handler>
    expected<db::RegData> getRegData(IoResultYield<Handler> yield) const {
        ExplainedError ec;
        auto regData = metaAdaptor.getRegData(yield[ec]);
        if (ec) {
            return make_unexpected(std::move(ec));
        }
        return regData;
    }

    std::optional<Shard::Id> getRandomShardIdWithAliveMaster(db::WeightedShardIds&& weightedShardIds) const {
        return sharpei::getRandomShardIdWithAliveMaster(*cache, std::move(weightedShardIds));
    }

    expected<Shard> getShardFromCache(const Shard::Id shardId) const {
        auto [ec, shard] = cache->getShard(shardId);
        if (ec) {
            return make_unexpected(std::move(ec));
        }
        return *shard;
    }

    void finish(const Shard& shard) const {
        using namespace ymod_webserver::helpers;
        using namespace ymod_webserver::helpers::transfer_encoding;
        using reflection::makeShardOrderedByState;
        const auto ordered = makeShardOrderedByState(shard);
        Response(*context.response).ok(fixed_size(format::json(ordered)));
    }

    void finish(const ExplainedError& error) const {
        using namespace ymod_webserver::helpers;
        using namespace ymod_webserver::helpers::transfer_encoding;
        if (error == Error::invalidRequest) {
            LOGDOG_(context.scribe.logger, warning, log::error_code=error);
            Response(*context.response).bad_request(fixed_size(format::json(error)));
        } else if (error == Error::domainNotFound) {
            LOGDOG_(context.scribe.logger, warning, log::error_code=error);
            Response(*context.response).not_found(fixed_size(format::json(error)));
        } else {
            LOGDOG_(context.scribe.logger, error, log::error_code=error);
            Response(*context.response).internal_server_error(fixed_size(format::json(error)));
        }
    }
};

} // namespace sharpei::server::handlers
