#pragma once

#include <boost/asio/spawn.hpp>

#include <internal/reflection/shard.h>
#include <internal/server/request_context.h>
#include <internal/db/adaptors/meta_adaptor.h>
#include <internal/db/adaptors/peers_adaptor.h>
#include <internal/config.h>
#include <internal/expected.h>
#include <internal/random.h>
#include <internal/random_shard_id_with_alive_master.h>

#include <ymod_webserver_helpers/helpers.hpp>

namespace sharpei::server::handlers {

using sharpei::server::RequestContext;

template <class MetaAdaptor, class PeersAdaptor>
class GetOrganizationPerformer {
public:
    GetOrganizationPerformer(RequestContext context, ConfigPtr config, MetaAdaptor metaAdaptor,
            PeersAdaptor peersAdaptor, cache::CachePtr cache)
        : context(std::move(context)),
          config(std::move(config)),
          metaAdaptor(std::move(metaAdaptor)),
          peersAdaptor(std::move(peersAdaptor)),
          cache(std::move(cache))
    {}

    void operator ()(boost::asio::yield_context yield) const {
        getOrgIdFromContext()
            .bind([&] (const auto orgId) { return getOrganizationShardId(orgId, yield); })
            .bind([&] (const auto shardId) { return getShardFromCache(shardId); })
            .bind([&] (const auto& shard) { return finish(shard); })
            .catch_error([&] (const auto& error) { return finish(error); });
    }

private:
    template <class T>
    using IoResultYield = io_result::basic_yield_context<T>;

    RequestContext context;
    ConfigPtr config;
    MetaAdaptor metaAdaptor;
    PeersAdaptor peersAdaptor;
    cache::CachePtr cache;

    expected<OrgId> getOrgIdFromContext() const {
        const auto& params = context.request->url.params;
        const auto orgId = params.find("org_id");
        if (orgId == params.end()) {
            return make_unexpected(ExplainedError(Error::invalidRequest, "org_id parameter not found"));
        }
        auto [ok, result] = lexicalCast<OrgId>(orgId->second);
        if (!ok) {
            return make_unexpected(ExplainedError(Error::invalidRequest, "invalid org_id parameter value"));
        }
        return result;
    }

    expected<Shard::Id> getOrganizationShardId(const OrgId orgId, boost::asio::yield_context yield) const {
        ExplainedError ec;
        const auto result = metaAdaptor.getOrganizationShardId(orgId, io_result::make_yield_context(yield)[ec]);
        if (!ec) {
            return result;
        } else if (ec != Error::organizationNotFound) {
            return make_unexpected(std::move(ec));
        } else {
            LOGDOG_(context.scribe.logger, notice, log::error_code=ec);
            return createOrganization(orgId, std::optional<DomainId>(), io_result::make_yield_context(yield)[ec]);
        }
    }

    template <class Handler>
    expected<Shard::Id> createOrganization(const OrgId orgId, const std::optional<DomainId>& domainId,
            IoResultYield<Handler> yield) const {
        return getMaster(yield).bind([&](auto masterHost) {
            return getRegData(yield).bind([&](auto regData) {
                return createOrganization(orgId, domainId, std::move(masterHost), std::move(regData), yield);
            });
        });
    }

    template <class Handler>
    expected<Shard::Id> createOrganization(const OrgId orgId,
                                           const std::optional<DomainId>& domainId,
                                           std::string masterHost,
                                           db::RegData regData,
                                           IoResultYield<Handler> yield) const {
        if (const auto shardId = getRandomShardIdWithAliveMaster(std::move(regData.weightedShardIds))) {
            db::CreateOrganizationParams params;
            params.orgId = orgId;
            params.domainId = domainId;
            params.shardId = *shardId;
            ExplainedError ec;
            const auto realShardId = peersAdaptor.createOrganization(masterHost, params, yield[ec]);
            if (ec) {
                return make_unexpected(std::move(ec));
            }
            if (*shardId == realShardId) {
                LOGDOG_(context.scribe.logger, notice, log::message="organization created", log::shard_id=realShardId);
            }
            return realShardId;
        }
        return make_unexpected(ExplainedError(Error::noShardWithAliveMaster));
    }

    template <class Handler>
    expected<std::string> getMaster(IoResultYield<Handler> yield) const {
        ExplainedError ec;
        auto masterHost = metaAdaptor.getMaster(yield[ec]);
        if (ec) {
            return make_unexpected(std::move(ec));
        }
        return masterHost;
    }

    template <class Handler>
    expected<db::RegData> getRegData(IoResultYield<Handler> yield) const {
        ExplainedError ec;
        auto regData = metaAdaptor.getRegData(yield[ec]);
        if (ec) {
            return make_unexpected(std::move(ec));
        }
        return regData;
    }

    std::optional<Shard::Id> getRandomShardIdWithAliveMaster(db::WeightedShardIds&& weightedShardIds) const {
        return sharpei::getRandomShardIdWithAliveMaster(*cache, std::move(weightedShardIds));
    }

    expected<Shard> getShardFromCache(const Shard::Id shardId) const {
        auto [ec, shard] = cache->getShard(shardId);
        if (ec) {
            return make_unexpected(std::move(ec));
        }
        return *shard;
    }

    void finish(const Shard& shard) const {
        using namespace ymod_webserver::helpers;
        using namespace ymod_webserver::helpers::transfer_encoding;
        using reflection::makeShardOrderedByState;
        const auto ordered = makeShardOrderedByState(shard);
        Response(*context.response).ok(fixed_size(format::json(ordered)));
    }

    void finish(const ExplainedError& error) const {
        using namespace ymod_webserver::helpers;
        using namespace ymod_webserver::helpers::transfer_encoding;
        if (error == Error::invalidRequest) {
            LOGDOG_(context.scribe.logger, warning, log::error_code=error);
            Response(*context.response).bad_request(fixed_size(format::json(error)));
        } else if (error == Error::organizationNotFound) {
            LOGDOG_(context.scribe.logger, warning, log::error_code=error);
            Response(*context.response).not_found(fixed_size(format::json(error)));
        } else {
            LOGDOG_(context.scribe.logger, error, log::error_code=error);
            Response(*context.response).internal_server_error(fixed_size(format::json(error)));
        }
    }
};

} // namespace sharpei::server::handlers
