#pragma once

#include <boost/any.hpp>
#include <boost/asio/coroutine.hpp>

#include <internal/config.h>
#include <internal/helpers.h>
#include <internal/db/adaptors/meta_adaptor.h>
#include <internal/db/adaptors/peers_adaptor.h>
#include <internal/server/handlers/base.h>
#include <internal/server/request_context.h>

#include <boost/asio/yield.hpp>

namespace sharpei {
namespace server {
namespace handlers {

template <class PeersAdaptorT, class UserIdValue>
class UpdateUserPerformer : public std::enable_shared_from_this<UpdateUserPerformer<PeersAdaptorT, UserIdValue>> {
public:
    using UserId = BasicUserId<UserIdValue>;
    using MetaAdaptorPtr = db::MetaAdaptorPtr<typename UserId::Value>;
    using PeersAdaptor = PeersAdaptorT;
    using RequestContext = sharpei::server::RequestContext;

    UpdateUserPerformer(
            RequestContext context,
            ConfigPtr config,
            MetaAdaptorPtr metaAdaptor,
            PeersAdaptor peersAdaptor);

    void perform();

private:
    RequestContext context_;
    ConfigPtr config_;
    MetaAdaptorPtr metaAdaptor_;
    PeersAdaptor peersAdaptor_;
    UserId uid_;
    boost::optional<Shard::Id> shardId_;
    boost::optional<Shard::Id> newShardId_;
    std::string data_;
    boost::asio::coroutine coroutine_;

    void continuation(ExplainedError error, boost::any args);

    void getMaster();
    void updateUser(const std::string& master);

    void finish(const ExplainedError& error) const;
};

template <class P, class U>
UpdateUserPerformer<P, U>::UpdateUserPerformer(RequestContext context,
    ConfigPtr config,
    MetaAdaptorPtr metaAdaptor,
    PeersAdaptor peersAdaptor)
        : context_(std::move(context)),
          config_(std::move(config)),
          metaAdaptor_(std::move(metaAdaptor)),
          peersAdaptor_(std::move(peersAdaptor)) {
}

template <class P, class U>
void UpdateUserPerformer<P, U>::perform()  {
    const auto& params = context_.request->url.params;
    const auto uid = params.find("uid");

    if (uid == params.end()) {
        return finish(ExplainedError(Error::invalidRequest, "uid parameter not found"));
    }

    const auto shardId = params.find("shard_id");
    const auto newShardId = params.find("new_shard_id");

    if (newShardId != params.end() && shardId == params.end()) {
        return finish(ExplainedError(Error::invalidRequest, "shard_id parameter not found but new_shard_id is present"));
    }

    if (!context_.request->raw_body.empty() && (context_.request->content.type != "application"
                                                || context_.request->content.subtype != "json")) {
        return finish(ExplainedError(Error::invalidRequest, "unsupported content type"));
    }

    bool ok;
    std::tie(ok, uid_) = lexicalCast<typename UserId::Value>(uid->second);
    if (!ok) {
        return finish(ExplainedError(Error::invalidRequest, "invalid uid parameter value"));
    }

    if (shardId != params.end()) {
        std::tie(ok, shardId_) = lexicalCast<Shard::Id>(shardId->second);
        if (!ok) {
            return finish(ExplainedError(Error::invalidRequest, "invalid shard_id parameter value"));
        }
    }

    if (newShardId != params.end()) {
        std::tie(ok, newShardId_) = lexicalCast<Shard::Id>(newShardId->second);
        if (!ok) {
            return finish(ExplainedError(Error::invalidRequest, "invalid new_shard_id parameter value"));
        }
    }

    data_ = std::string(context_.request->raw_body.begin(), context_.request->raw_body.end());

    continuation(ExplainedError(Error::ok), boost::any());
}

template <class P, class U>
void UpdateUserPerformer<P, U>::continuation(ExplainedError error, boost::any args) {
    try {
        if (error) {
            return finish(std::move(error));
        }
        reenter(coroutine_) {
            yield getMaster();
            yield updateUser(boost::any_cast<std::string&>(args));
            finish(std::move(error));
        }
    } catch (const boost::exception& exception) {
        finish(ExplainedError(Error::internalError, boost::diagnostic_information(exception)));
    } catch (const std::exception& exception) {
        finish(ExplainedError(Error::internalError, exception.what()));
    } catch (...) {
        finish(ExplainedError(Error::internalError, "unknown error"));
    }
}

template <class P, class U>
void UpdateUserPerformer<P, U>::getMaster() {
    const auto self = this->shared_from_this();
    auto onResult = [self] (auto value) { self->continuation(ExplainedError(Error::ok), std::move(value)); };
    auto onError = [self] (auto error) { self->continuation(std::move(error), boost::any()); };
    metaAdaptor_->getMaster(std::move(onResult), std::move(onError));
}

template <class P, class U>
void UpdateUserPerformer<P, U>::updateUser(const std::string& master) {
    db::UpdateUserParams<typename UserId::Value> params {
        uid_,
        shardId_,
        newShardId_,
        data_.empty() ? boost::none : boost::optional<std::string>(std::move(data_))
    };
    const auto self = this->shared_from_this();
    peersAdaptor_.updateUser(master, std::move(params),
        [self] (auto error) { self->continuation(std::move(error), boost::any()); });
}

template <class P, class U>
void UpdateUserPerformer<P, U>::finish(const ExplainedError& error) const {
    using namespace ymod_webserver::helpers;
    using namespace ymod_webserver::helpers::transfer_encoding;
    if (!error) {
        Response(*context_.response).ok(fixed_size(format::json(std::string("done"))));
    } else if (error == Error::invalidRequest || error == Error::invalidUserShardId) {
        LOGDOG_(context_.scribe.logger, warning, log::error_code=error);
        Response(*context_.response).bad_request(fixed_size(format::json(error)));
    } else if (error == Error::uidNotFound) {
        LOGDOG_(context_.scribe.logger, warning, log::error_code=error);
        Response(*context_.response).not_found(fixed_size(format::json(error)));
    } else {
        LOGDOG_(context_.scribe.logger, error, log::error_code=error);
        Response(*context_.response).internal_server_error(fixed_size(format::json(error)));
    }
}

} // namespace handlers
} // namespace server
} // namespace sharpei

#include <boost/asio/unyield.hpp>
