#pragma once

#include <internal/cache/cache.h>
#include <internal/db/adaptors/meta_adaptor.h>
#include <internal/db/adaptors/meta_adaptor_factory.h>
#include <internal/db/adaptors/shard_adaptor.h>
#include <internal/db/conn_info.h>
#include <internal/helpers.h>
#include <internal/reflection/shard.h>
#include <internal/server/handlers/base.h>

#include <yamail/data/serialization/json_writer.h>
#include <yplatform/find.h>

#include <boost/algorithm/cxx11/copy_if.hpp>
#include <boost/algorithm/string/join.hpp>
#include <boost/asio/coroutine.hpp>
#include <boost/range/adaptor/transformed.hpp>
#include <boost/range/algorithm/copy.hpp>

namespace sharpei {
namespace server {
namespace handlers {

using sharpei::server::RequestContext;
using sharpei::server::handlers::Base;

struct ResponseShard {
    void operator ()(ymod_webserver::response_ptr response, const reflection::ShardWithoutRoles& shard, const boost::optional<std::string>&) const {
        using namespace ymod_webserver::helpers;
        using namespace ymod_webserver::helpers::transfer_encoding;
        return Response(*response).ok(fixed_size(format::json(shard)));
    }

    void operator ()(ymod_webserver::response_ptr response, const reflection::Shard& shard, const boost::optional<std::string>&) const {
        using namespace ymod_webserver::helpers;
        using namespace ymod_webserver::helpers::transfer_encoding;
        return Response(*response).ok(fixed_size(format::json(shard)));
    }
};

template <class ErrorHandlerT, class ResponseHandlerT, class UserIdValueT, db::UserType userType = db::UserType::existing, db::QueryType queryType = db::QueryType::withData>
class GetUserAsyncHandler: public MakeSharedFromThis<GetUserAsyncHandler<ErrorHandlerT, ResponseHandlerT, UserIdValueT, userType, queryType>> {
public:
    using ErrorHandler = ErrorHandlerT;
    using ResponseHandler = ResponseHandlerT;
    using UserId = BasicUserId<UserIdValueT>;
    using MetaAdaptorPtr = sharpei::db::MetaAdaptorPtr<typename UserId::Value>;
    using Availability = cache::Availability;

    struct Params {
        UserId uid;
        Mode mode;
        bool isForced;
    };

    GetUserAsyncHandler(const RequestContext& req,
                        ConfigPtr config,
                        cache::CachePtr cache,
                        db::MetaPoolPtr metaPool,
                        db::ShardPoolPtr shardPool,
                        db::MetaAdaptorFactoryPtr metaAdaptorFactory,
                        boost::asio::io_service& ioService,
                        ErrorHandler errorHandler = ErrorHandler(),
                        ResponseHandler responseHandler = ResponseHandler())
        : req_(req),
          config_(config),
          cache_(cache),
          shardPool_(shardPool),
          metaAdaptor_(metaAdaptorFactory->getMetaAdaptor<typename UserId::Value>(
              config->meta.db.adaptor, metaPool, req_.scribe)),
          ioService_(ioService),
          errorHandler_(std::move(errorHandler)),
          responseHandler_(std::move(responseHandler)) {
    }

    void execute() {
        params_ = getParams(req_);
        auto ptr = this->shared_from_this();
        auto safeHandler = [ptr] (auto shardId, auto data) {
            SHARPEI_SAFE_EXEC(ptr->req_, ([&] { ptr->onGetUserData(shardId, std::move(data)); }));
        };
        auto errorHandler = [ptr] (const ExplainedError& error) {
            SHARPEI_SAFE_EXEC(ptr->req_, ([&] { ptr->onError(error); }));
        };
        getUserData(std::move(safeHandler), std::move(errorHandler));
    }

    const RequestContext& req() const {
        return req_;
    }

    const Params& params() const {
        return params_;
    }

    ConfigPtr config() const {
        return config_;
    }

    db::ShardPoolPtr shardPool() const {
        return shardPool_;
    }

    MetaAdaptorPtr metaAdaptor() const {
        return metaAdaptor_;
    }

    cache::CachePtr cache() const {
        return cache_;
    }

    boost::asio::coroutine& coroutine() {
        return coroutine_;
    }

    Params getParams(const RequestContext& req) const {
        Params res;
        res.uid = req.getRequiredArg<typename UserId::Value>("uid");
        const std::string modeStr = req.getRequiredArg("mode");
        res.mode = Mode(modeStr);
        if (res.mode == Mode::Unknown) {
            throw ParamsException("invalid mode value: " + modeStr);
        }
        res.isForced = !req.getArg("force").empty();
        return res;
    }

    void onGetUserData(Shard::Id shardId, boost::optional<std::string> data) {
        const auto roles = makeRoles(params_.mode);
        if (roles.empty()) {
            std::string msg = fmt::format("No roles for mode {} {}", params_.mode.toString(), shardId);
            onError(ExplainedError(Error::modeRolesNotFound, msg));
            return;
        }
        const auto availabilityMethod = params_.isForced ? &Availability::recent
                                                         : &Availability::smooth;
        const auto shardName = cache_->shardName.get(shardId);
        if (!shardName) {
            std::string msg = fmt::format("No entry in cache: could not find name for shard {}", shardId);
            onError(ExplainedError(Error::shardNotFound, msg));
            return;
        }
        const auto databases = getDatabasesWithRoles(shardId, roles, availabilityMethod);
        if (databases.required.empty()) {
            const auto error = makeErrorMessage(roles, databases.all);
            onError(ExplainedError(Error::appropriateHostNotFound, error));
            return;
        }
        onGetShard(Shard(shardId, *shardName, databases.required), data);
    }

    void onGetShard(const Shard& shard, const boost::optional<std::string>& data) {
        if (shard.databases.empty()) {
            std::string msg = fmt::format("Could not find appropriate hosts for shard {}", shard.id);
            onError(ExplainedError(Error::appropriateHostNotFound, msg));
            return;
        }
        switch (params_.mode.value) {
            case Mode::Master:
            case Mode::Replica:
                return response(reflection::makeShardWithoutRolesOrderedByState(shard), data);
            case Mode::ReadWrite:
            case Mode::WriteRead:
                return response(reflection::makeShardOrderedByModeAndState(shard, params_.mode), data);
            default:
                return response(reflection::makeShardOrderedByState(shard), data);
        }
    }

    void response(const reflection::ShardWithoutRoles& shard, const boost::optional<std::string>& data) {
        responseHandler_(req_.response, shard, data);
    }

    void response(const reflection::Shard& shard, const boost::optional<std::string>& data) {
        responseHandler_(req_.response, shard, data);
    }

    void onError(const ExplainedError& error) {
        errorHandler_(this->shared_from_this(), error);
    }

    static std::set<Shard::Database::Role> makeRoles(Mode mode) {
        using Role = Shard::Database::Role;
        switch (mode.value) {
            case Mode::Master:
            case Mode::WriteOnly:
                return {Role::Master};
            case Mode::Replica:
            case Mode::ReadOnly:
                return {Role::Replica};
            case Mode::All:
            case Mode::ReadWrite:
            case Mode::WriteRead:
                return {Role::Master, Role::Replica};
            default:
                return {};
        }
    }

    bool needOnlyAlive() const {
        return  params_.mode == Mode::Master || params_.mode == Mode::Replica;
    }

    struct DatabasesWithRoles {
        Shard::Databases all;
        Shard::Databases required;
    };

    DatabasesWithRoles getDatabasesWithRoles(const Shard::Id& shardId,
                                             const std::set<Shard::Database::Role>& roles,
                                             Availability::GetMethod availabilityMethod) const {
        using Db = Shard::Database;
        using Status = Shard::Database::Status;
        using boost::algorithm::copy_if;
        DatabasesWithRoles result;
        const auto hosts = cache_->role.get(shardId);
        if (!hosts.empty()) {
            LOGDOG_(req_.scribe.logger, debug, log::message="cache hit", log::shard_id=shardId);
            result.all = makeShardDatabases(shardId, hosts, availabilityMethod, *cache_);
            for (const auto& role : roles) {
                if (needOnlyAlive()) {
                    copy_if(result.all, std::inserter(result.required, result.required.end()),
                            [&] (const Db& x) {
                                return x.role() == role && x.status() == Status::Alive;
                            });
                } else {
                    copy_if(result.all, std::inserter(result.required, result.required.end()),
                            [&] (const Db& x) { return x.role() == role; });
                }
            }
        }
        return result;
    }

    static std::string makeErrorMessage(const std::set<Shard::Database::Role>& roles,
                                        const Shard::Databases& databases) {
        using Db = Shard::Database;
        using Role = Db::Role;
        using boost::adaptors::transformed;
        using boost::join;
        std::ostringstream error;
        error << "No entry in cache: ";
        if (databases.empty()) {
            if (roles.size() > 1) {
                error << "databases with roles "
                      << join(roles | transformed(std::mem_fn(&Role::toString)), " and ")
                      << " not found";
            } else {
                error << "databases with role " << roles.begin()->toString() << " not found";
            }
        } else {
            error << join(databases | transformed([] (const Db& x) {
                return x.address().host + " is " + x.status().toString() + " " + x.role().toString();
            }), ", ");
        }
        return error.str();
    }

    template <class Handler>
    auto wrap(Handler handler) {
        return ioService_.wrap(std::move(handler));
    }

private:
    RequestContext req_;
    ConfigPtr config_;
    cache::CachePtr cache_;
    db::ShardPoolPtr shardPool_;
    MetaAdaptorPtr metaAdaptor_;
    boost::asio::io_service& ioService_;
    ErrorHandler errorHandler_;
    ResponseHandler responseHandler_;
    Params params_;
    boost::asio::coroutine coroutine_;

    template <class SafeHandler, class ErrorHandler>
    void getUserData(SafeHandler safeHandler, ErrorHandler errorHandler) const {
        if constexpr (userType == db::UserType::existing) {
            metaAdaptor_->getUserData(params_.uid, safeHandler, errorHandler, queryType);
        } else {
            metaAdaptor_->getDeletedUserData(params_.uid, safeHandler, errorHandler, queryType);
        }
    }

};

template <class ErrorHandlerT,
          class ResponseHandlerT,
          class UserIdValueT,
          db::UserType userType = db::UserType::existing,
          db::QueryType queryType = db::QueryType::withData>
class BaseGetUser : public Base {
public:
    using ErrorHandler = ErrorHandlerT;
    using ResponseHandler = ResponseHandlerT;
    using UserId = BasicUserId<UserIdValueT>;

    BaseGetUser(ConfigPtr config,
                cache::CachePtr cache,
                db::MetaPoolPtr metaPool,
                db::ShardPoolPtr shardPool,
                db::MetaAdaptorFactoryPtr metaAdaptorFactory,
                ErrorHandler errorHandler = ErrorHandler(),
                ResponseHandler responseHandler = ResponseHandler())
        : config(config),
          cache(cache),
          metaPool(metaPool),
          shardPool(shardPool),
          metaAdaptorFactory(metaAdaptorFactory),
          errorHandler_(std::move(errorHandler)),
          responseHandler_(std::move(responseHandler)) {
    }

    ymod_webserver::methods::http_method method() const override final {
        return ymod_webserver::methods::mth_get;
    }

    void execute(RequestContext& req) const override final {
        using AsyncHandler = GetUserAsyncHandler<ErrorHandler, ResponseHandler, typename UserId::Value, userType, queryType>;
        const auto io = yplatform::global_net_reactor->get_pool()->io();
        auto asyncHandler = AsyncHandler::create(req,
                                                 config,
                                                 cache,
                                                 metaPool,
                                                 shardPool,
                                                 metaAdaptorFactory,
                                                 *io,
                                                 errorHandler_,
                                                 responseHandler_);
        asyncHandler->execute();
    }

private:
    ConfigPtr config;
    cache::CachePtr cache;
    db::MetaPoolPtr metaPool;
    db::ShardPoolPtr shardPool;
    db::MetaAdaptorFactoryPtr metaAdaptorFactory;
    ErrorHandler errorHandler_;
    ResponseHandler responseHandler_;
};

struct DefaultErrorHandler {
    template <class AsyncHandler>
    void operator ()(std::shared_ptr<AsyncHandler> asyncHandler, const ExplainedError& error) const {
        using namespace ymod_webserver::helpers;
        using namespace ymod_webserver::helpers::transfer_encoding;
        if (error == Error::uidNotFound) {
            LOGDOG_(asyncHandler->req().scribe.logger, error, log::error_code=error);
            Response(*asyncHandler->req().response).not_found(fixed_size(format::json(ExplainedResponse(error))));
        } else {
            LOGDOG_(asyncHandler->req().scribe.logger, error, log::error_code=error);
            Response(*asyncHandler->req().response).internal_server_error(fixed_size(format::json(ExplainedResponse(error))));
        }
    }
};

} // namespace handlers
} // namespace server
} // namespace sharpei
