#pragma once

#include <internal/db/adaptors/meta_adaptor.h>
#include <internal/db/adaptors/meta_adaptor_factory.h>
#include <internal/db/adaptors/shard_adaptor.h>
#include <internal/helpers.h>
#include <internal/server/handlers/base.h>

namespace sharpei {
namespace server {
namespace handlers {

class Reset : public Base {
public:
    Reset(ConfigPtr config,
          cache::CachePtr cache,
          db::MetaPoolPtr metaPool,
          db::ShardPoolPtr shardPool,
          db::MetaAdaptorFactoryPtr metaAdaptorFactory)
        : config_(config),
          cache_(cache),
          metaPool_(metaPool),
          shardPool_(shardPool),
          metaAdaptorFactory_(metaAdaptorFactory) {
    }

private:
    class AsyncHandler: public MakeSharedFromThis<AsyncHandler> {
    public:
        void execute() {
            auto shardId = getShardIdParam(req_);
            cache_->role.erase(shardId);
            cache_->status.erase(shardId);
            cache_->state.erase(shardId);
            auto metaAdaptor = metaAdaptorFactory_->getBaseMetaAdaptor(
                config_->meta.db.adaptor, metaPool_, req_.scribe);
            auto ptr = shared_from_this();
            auto safeHandler = [ptr] (const ShardWithoutRoles& shard) {
                SHARPEI_SAFE_EXEC(ptr->req_, ([&] { ptr->onGetShard(shard); }));
            };
            auto errorHandler = [ptr] (const ExplainedError& error) {
                SHARPEI_SAFE_EXEC(ptr->req_, ([&] { ptr->onError(error); }));
            };
            metaAdaptor->getShard(shardId, safeHandler, errorHandler);
        }

    private:
        friend class EnableStaticConstructor<AsyncHandler>;

        AsyncHandler(const RequestContext& req,
                     const ConfigPtr& config,
                     const cache::CachePtr& cache,
                     const db::MetaPoolPtr& metaPool,
                     const db::ShardPoolPtr& shardPool,
                     db::MetaAdaptorFactoryPtr metaAdaptorFactory)
            : req_(req),
              config_(config),
              cache_(cache),
              metaPool_(metaPool),
              shardPool_(shardPool),
              metaAdaptorFactory_(metaAdaptorFactory) {
        }

        static Shard::Id getShardIdParam(const RequestContext& req) {
            return req.getRequiredArg<Shard::Id>("shard");
        }

        void onGetShard(const ShardWithoutRoles& shard) {
            using db::ShardAdaptorPtr;
            using db::getShardAdaptor;
            ShardAdaptorPtr shardAdaptor = getShardAdaptor(config_->shard.adaptor, shardPool_, req_.scribe, cache_);
            auto ptr = shared_from_this();
            auto safeHandler = [ptr] () {
                SHARPEI_SAFE_EXEC(ptr->req_, ([&] { ptr->onFinish(); }));
            };
            auto errorHandler = [ptr] (const ExplainedError& error) {
                SHARPEI_SAFE_EXEC(ptr->req_, ([&] { ptr->onError(error); }));
            };
            shardAdaptor->resetHostCache(shard, safeHandler, errorHandler);
        }

        void onError(const ExplainedError& error) {
            LOGDOG_(req_.scribe.logger, error, log::error_code=error);
            const auto message = error.full_message();
            if (error == Error::shardNotFound) {
                Response(*req_.response).not_found(fixed_size(format::text(message)));
            } else if (error == Error::invalidUserShardId) {
                Response(*req_.response).bad_request(fixed_size(format::text(message)));
            } else {
                Response(*req_.response).internal_server_error(fixed_size(format::text(message)));
            }
        }

        void onFinish() {
            Response(*req_.response).ok(fixed_size(format::text("ok")));
        }

        RequestContext req_;
        ConfigPtr config_;
        cache::CachePtr cache_;
        db::MetaPoolPtr metaPool_;
        db::ShardPoolPtr shardPool_;
        db::MetaAdaptorFactoryPtr metaAdaptorFactory_;
    };

    ymod_webserver::methods::http_method method() const override final {
        return ymod_webserver::methods::mth_post;
    }

    void execute(RequestContext& req) const override final {
        AsyncHandler::create(req, config_, cache_, metaPool_, shardPool_, metaAdaptorFactory_)
            ->execute();
    }

private:
    ConfigPtr config_;
    cache::CachePtr cache_;
    db::MetaPoolPtr metaPool_;
    db::ShardPoolPtr shardPool_;
    db::MetaAdaptorFactoryPtr metaAdaptorFactory_;
};

} // namespace handlers
} // namespace server
} // namespace sharpei
