#ifndef SHARPEI_SERVER_REQUEST_CONTEXT_H
#define SHARPEI_SERVER_REQUEST_CONTEXT_H

#include <mutex>
#include <ymod_webserver/server.h>
#include <internal/errors.h>
#include <internal/logger.h>
#include <internal/helpers.h>
#include <boost/algorithm/string/case_conv.hpp>

namespace sharpei {
namespace server {

struct RequestContext {
    ymod_webserver::request_ptr request;
    ymod_webserver::response_ptr response;
    Scribe scribe;
    std::string method;

    RequestContext(ymod_webserver::request_ptr request, ymod_webserver::response_ptr response, std::string method)
        : request(std::move(request))
        , response(std::move(response))
        , scribe(sessionId(), requestId())
        , method(std::move(method))
    {}

    std::string getArg(const std::string& name) const {
        auto iter = request->url.params.find(name);
        return iter == request->url.params.end() ? "" : iter->second;
    }

    template <typename T>
    T getArg(const std::string& name, const T& def = T()) const {
        std::string arg = getArg(name);
        if (arg.empty()) {
            return def;
        }
        auto [ok, result] = lexicalCast<T>(arg);
        if (!ok) {
            throw ParamsException("invalid argument for " + name);
        }
        return result;
    }

    std::string getRequiredArg(const std::string& name) const {
        const std::string arg = getArg(name);
        if (arg.empty()) {
            throw ParamsException(name + " must not be empty");
        }
        return arg;
    }

    template <typename T>
    T getRequiredArg(const std::string& name) const {
        const std::string arg = getArg(name);
        if (arg.empty()) {
            throw ParamsException(name + " must not be empty");
        }
        auto [ok, result] = lexicalCast<T>(arg);
        if (!ok) {
            throw ParamsException("invalid argument for " + name);
        }
        return result;
    }

    template <typename T>
    std::optional<T> getOptionalArg(const std::string& name) const {
        if (request->url.params.find(name) != request->url.params.end()) {
            return getRequiredArg<T>(name);
        } else {
            return std::nullopt;
        }
    }

    std::string getHeader(const std::string& name) const {
        auto iter = request->headers.find(boost::algorithm::to_lower_copy(name));
        return iter == request->headers.end() ? "" : iter->second;
    }

    std::string sessionId() const {
        return request->ctx()->uniq_id();
    }

    std::string requestId() const {
        return getHeader("X-Request-Id");
    }

    std::string userIp() const {
        const auto it = request->headers.find("x-real-ip");
        return it == request->headers.end() || it->second.empty()
            ? request->context->remote_address : it->second;
    }
};

} // namespace server
} // namespace sharpei

#endif // SHARPEI_SERVER_REQUEST_CONTEXT_H
