#pragma once

#include <internal/services/blackbox/blackbox.h>
#include <internal/services/blackbox/blackbox_config.h>
#include <internal/services/types.h>
#include <internal/errors.h>

#include <butil/http/headers.h>

#include <sstream>

namespace sharpei::services::blackbox {

class BlackboxImpl : public Blackbox {
public:
    BlackboxImpl(BlackboxConfig config, std::shared_ptr<ymod_httpclient::cluster_call> httpClient);

    expected<std::vector<HostedDomain>> getHostedDomains(DomainId domainId,
            const TaskContextPtr& context) const override;

    expected<std::vector<HostedDomain>> getHostedDomains(const std::string& domain,
            const TaskContextPtr& context) const override;

private:
    BlackboxConfig config;
    ymod_httpclient::cluster_client_options httpClientOptions;
    std::shared_ptr<ymod_httpclient::cluster_call> httpClient;

    template <class T>
    struct WriteParam {
        static_assert(std::is_void_v<T>, "unsupported blackbox parameter");
    };

    template <>
    struct WriteParam<DomainId> {
        DomainId value;

        friend std::ostream& operator <<(std::ostream& stream, const WriteParam& wrapper) {
            return stream << "domain_id=" << wrapper.value;
        }
    };

    template <>
    struct WriteParam<std::string> {
        const std::string& value;

        friend std::ostream& operator <<(std::ostream& stream, const WriteParam& wrapper) {
            return stream << "domain=" << wrapper.value;
        }
    };

    template <class T>
    expected<std::vector<HostedDomain>> getHostedDomainsImpl(const T& param, const TaskContextPtr& context) const {
        std::string url = fmt::format("?method=hosted_domains&format=json&{}", WriteParam<std::decay_t<T>> {param});
        http::headers headers;
        headers.add("X-Request-Id", context->requestId());

        const auto request = HttpRequest::GET(url, headers.format());

        boost::system::error_code ec;
        const auto response = httpClient->async_run(
            context,
            request, 
            config.http.options,
            context->yield()[ec]
        );

        if (ec) {
            LOGDOG_(context->scribe().logger, error, log::url=request.url, log::error_code=ec);
            return make_unexpected(ExplainedError(std::move(ec)));
        }

        if (response.status != 200) {
            LOGDOG_(context->scribe().logger, error, log::url=request.url, log::body=response.body);
            return make_unexpected(ExplainedError(Error::blackBoxHttpError));
        }

        return parse(response.body, context);
    }

    expected<std::vector<HostedDomain>> parse(const std::string& body, const TaskContextPtr& context) const;
};

} // namespace sharpei::services::blackbox
