#pragma once

#include <string>
#include <vector>
#include <set>
#include <unordered_map>
#include <boost/bind.hpp>
#include <boost/optional.hpp>
#include <internal/mode.h>

namespace sharpei {

class Shard {
public:
    typedef unsigned int Id;
    typedef std::string Name;

    class Database {
    public:
        struct Role {
            enum Value {
                Master,
                Replica
            } value;

            Role(Value value)
                : value(value)
            {}

            Role(const std::string& str);

            static Value valueFromString(const std::string& str);
            std::string toString() const;

            operator Value() const {
                return value;
            }
        };

        struct Status {
            enum Value {
                Alive,
                Dead
            } value;

            Status(Value value)
                : value(value) {}

            operator Value() const {
                return value;
            }

            std::string toString() const;
        };

        struct Address {
            std::string host;
            unsigned port;
            std::string dbname;
            std::string dataCenter;

            Address()
                : port(0)
            {}

            Address(const std::string& host, unsigned port, const std::string& dbname, const std::string& dataCenter)
                : host(host), port(port), dbname(dbname), dataCenter(dataCenter)
            {}

            bool operator<(const Address& other) const {
                if (host != other.host) {
                    return host < other.host;
                } else if (port != other.port) {
                    return port < other.port;
                } else if (dbname != other.dbname) {
                    return dbname < other.dbname;
                } else {
                    return dataCenter < other.dataCenter;
                }
            }

            bool operator==(const Address& other) const {
                return host == other.host && port == other.port && dbname == other.dbname
                    && dataCenter == other.dataCenter;
            }

            bool operator!=(const Address& other) const {
                return !(*this == other);
            }
        };

        struct State {
            using ReplicationLag = std::int32_t;

            ReplicationLag lag;

            bool operator==(const State& other) const {
                return lag == other.lag;
            }

            bool operator<(const State& other) const {
                return lag < other.lag;
            }
        };

        Database(const Address& address, Role role, Status status, State state)
            : address_(address), role_(role), status_(status), state_(std::move(state))
        {}

        const Address& address() const {
            return address_;
        }

        Role role() const {
            return role_;
        }

        Status status() const {
            return status_;
        }

        const State& state() const {
            return state_;
        }

        bool operator==(const Database& other) const {
            return address_ == other.address_ && role_ == other.role_ && status_ == other.status_ &&
                    state_ == other.state_;
        }

    private:
        Address address_;
        Role role_;
        Status status_;
        State state_;
    };

    struct LessByAddress {
        bool operator()(const Database& lhs, const Database& rhs) const {
            return lhs.address() < rhs.address();
        }
    };

    typedef std::set<Database, LessByAddress> Databases;

    Id id;
    Name name;
    Databases databases;

    Shard()
        : id(0)
    {}

    Shard(Id id, const Name& name, const Databases& databases)
        : id(id), name(name), databases(databases)
    {}
};

struct ShardWithoutRoles {
    typedef std::vector<Shard::Database::Address> Addresses;

    Shard::Id id {};
    Shard::Name name;
    Addresses addrs;

    ShardWithoutRoles() = default;
    ShardWithoutRoles(const Shard& shard)
        : id(shard.id), name(shard.name), addrs(makeAddrs(shard.databases)) {}
    ShardWithoutRoles(Shard::Id id, const Shard::Name& name, Addresses addrs)
        : id(id), name(name), addrs(std::move(addrs)) {}

    static Addresses makeAddrs(const Shard::Databases& databases) {
        Addresses result;
        for (const auto& db : databases) {
            result.push_back(db.address());
        }
        std::sort(result.begin(), result.end());
        result.erase(std::unique(result.begin(), result.end()), result.end());
        return result;
    }
};

using ShardsInfoNewFormat = std::unordered_map<Shard::Id, Shard>;
using ShardsInfoOldFormat = std::unordered_map<Shard::Id, Shard::Databases>;

} // namespace sharpei
