import argparse
import itertools
import json
import os.path
import random
import re
import requests
import sys

from itertools import cycle


def weighted_random(handlers):
	choices = [[handler] * weight for handler, weight in handlers.items()]
	return random.choice(list(itertools.chain.from_iterable(choices)))


def get_source(url, dst):
    session = requests.session()
    try:
        responce = session.get(url, stream=True)
        if responce.status_code != 200:
            raise Exception("Error during download file")
        with open(dst, 'wb') as resource:
            resource.write(responce.content)
        return os.path.abspath(dst)
    except Exception as ex:
        print("Can't download resource. {}".format(ex))
        sys.exit(2)
    finally:
        session.close()


class Ammo(object):
    def __init__(self, source, profile, count):
        if re.match(r'^http*', source):
            self.source = get_source(source, "uids.src")
        else:
            print("Unknown source link")
            sys.exit(1)
        self.ammo_count = count
        self.tags = [weighted_random(json.loads(profile.strip())) for _ in xrange(self.ammo_count)]
        self.legal_tags = (
            "conninfo_all",
            "conninfo_read",
            "conninfo_write",
            "conninfo_force",
            "registration_conninfo",
            "stat_v1",
            "stat_v2",
            "stat_v3"
        )


    def make_ammo(self):
        with open(self.source, "r") as src:
            uids = [line.strip() for line in src]
        uid = cycle(uids)
        bullits = []
        if len(uids) > 0:
            for tag in self.tags:
                if tag in self.legal_tags:
                    if tag in ("conninfo_withregistration", "stat_v1", "stat_v2", "stat_v3"):
                        bullit = {'tag': tag, 'uid': ""}
                    else:
                        bullit = {'tag': tag, 'uid': next(uid)}
                    bullits.append(bullit)
                else:
                    print("Wrong tag: {}".format(tag))
                    sys.exit(3)
        with open("ammo.json", "w") as ammo:
            ammo.write("\n".join(json.dumps(bullit) for bullit in bullits).encode('utf-8'))


if __name__ == "__main__":

    parser = argparse.ArgumentParser()
    parser.add_argument("--src", type=str, help="Link to datasource file")
    parser.add_argument("--profile", type=str, help="string with json profile, in handler:weight format", default='{"conninfo_all": 8, "stat_v2": 1, "conninfo_withregistration": 1}')
    parser.add_argument("--count", type=int, help="Count of bullits in ammo file", default=40000)
    args = parser.parse_args()

    ammo = Ammo(args.src, args.profile, args.count)
    ammo.make_ammo()
