#!/usr/bin/env python3

import json
import math

from argparse import ArgumentParser
from subprocess import Popen, PIPE, DEVNULL


SIGNALS = {
    "4xx": "access_log4xx_summ",
    "5xx": "access_log5xx_summ",
    "total": "access_log_summ",
}


def process_error_codes(metrics: dict, max_errors_perc: int):
    for error_type in ("4xx", "5xx"):
        try:
            total_requests = next(v[1] for v in metrics if v[0] == SIGNALS["total"])
            threshold = math.ceil((total_requests * max_errors_perc) // 100)
            errors = next(v[1] for v in metrics if v[0] == SIGNALS[error_type])
            print(
                "PASSIVE-CHECK:sharpei_"
                + error_type
                + ";%d;" % (2 if errors > threshold else 0)
            )
        except StopIteration:
            # absence of the corresponding signal means that there are no such status codes in logs at all
            print("PASSIVE-CHECK:sharpei_" + error_type + ";0;")


def parse_args():
    parser = ArgumentParser()
    parser.add_argument(
        "--max_errors_perc",
        default=1,
        type=int,
        choices=range(1, 100),
        help="maximum allowed percentage of errors",
    )
    return parser.parse_args()


try:
    args = parse_args()
    curl = Popen(
        [
            "curl",
            "--silent",
            "--verbose",
            "--max-time",
            "1",
            "--user-agent",
            "sharpei-check",
            "http://localhost:26010/unistat",
        ],
        stdout=PIPE,
        stderr=PIPE,
    )
    curl.wait(1)
    metrics = json.loads(curl.stdout.read().decode("utf-8"))
    process_error_codes(metrics, args.max_errors_perc)
except Exception as e:
    print(
        "PASSIVE-CHECK:sharpei_monitor;2;%s"
        % str(e).replace("\n", "\\n").replace("\r", "\\r").replace(";", "_")
    )
