#include <boost/range/adaptor/transformed.hpp>
#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/algorithm/copy.hpp>
#include <boost/range/algorithm/transform.hpp>
#include <internal/cache/cache.h>
#include <spdlog/details/format.h>


namespace sharpei {
namespace cache {

Cache::Cache(std::size_t historyCapacity, std::size_t errorsLimit, const Cache& other)
        : status(historyCapacity, errorsLimit) {
    const auto shards = makeShardsInfoOldFormat(other, &Availability::recent);
    for (const auto& shard : shards) {
        const auto& shardId = shard.first;
        const auto& databases = shard.second;
        for (const auto& database : databases) {
            switch (database.status()) {
                case Shard::Database::Status::Alive:
                    status.alive(shardId, database.address());
                    break;
                case Shard::Database::Status::Dead:
                    status.dead(shardId, database.address());
                    break;
            }
        }
    }
    role = other.role;
    state = other.state;
}

Cache::OptShardMaster Cache::getAliveShardMaster(Shard::Id shardId) const {
    const auto hosts = role.get(shardId, Shard::Database::Role::Master);
    if (hosts.empty()) {
        return boost::none;
    }
    const auto master = *hosts.begin();
    const auto hostStatus = status.get(shardId, master, &Availability::recent);
    if (!hostStatus.is_initialized() || hostStatus.get() != Shard::Database::Status::Alive) {
        return boost::none;
    }
    return master;
}

std::pair<ExplainedError, boost::optional<Shard>> Cache::getShard(Shard::Id shardId) const {
    const auto name = shardName.get(shardId);
    if (!name) {
        std::string error = fmt::format("for shard {}", shardId);
        return {ExplainedError(Error::noCachedShardName, error), boost::optional<Shard>()};
    }
    const auto databasesWithRoles = role.get(shardId);
    if (databasesWithRoles.empty()) {
        std::string error = fmt::format("for shard {}", shardId);
        return {ExplainedError(Error::noCachedShardDatabasesRoles, error), boost::optional<Shard>()};
    }
    const auto databases = makeShardDatabases(shardId, databasesWithRoles, &Availability::smooth, *this);
    if (databases.empty()) {
        std::string error = fmt::format("for shard {}", shardId);
        return {ExplainedError(Error::noCachedShardDatabases, error), boost::optional<Shard>()};
    }
    return {ExplainedError(Error::ok), Shard(shardId, name.get(), databases)};
}

yamail::expected<RoleCache::Shards> selectShards(const Cache& cache, std::optional<Shard::Id> shardId) {
    if (!shardId.has_value()) {
        return cache.role.all();
    } else {
        auto shard = cache.role.get(*shardId);
        if (!shard.empty()) {
            return RoleCache::Shards{{*shardId, std::move(shard)}};
        } else {
            std::string msg = fmt::format("shard with shard_id={} doesn't exist", *shardId);
            return yamail::make_unexpected(ExplainedError(Error::shardNotFound, msg));
        }
    }
}

ShardsInfoNewFormat makeShardsInfoNewFormat(const RoleCache::Shards& shards, const Cache& cache, Availability::GetMethod availability) {
    ShardsInfoNewFormat result;
    const auto names = cache.shardName.all();

    boost::transform(shards, std::inserter(result, result.end()),
        [&] (const auto& shard) -> ShardsInfoNewFormat::value_type {
            return {
                shard.first,
                Shard(shard.first, names.at(shard.first), makeShardDatabases(shard.first, shard.second, availability, cache))
            };
        });
    return result;
}

ShardsInfoOldFormat makeShardsInfoOldFormat(const Cache& cache, Availability::GetMethod availability) {
    ShardsInfoOldFormat result;
    using Ref = const RoleCache::Shards::value_type &;

    boost::transform(cache.role.all(), std::inserter(result, result.end()),
        [&] (Ref shard) -> ShardsInfoOldFormat::value_type {
            return {
                shard.first,
                makeShardDatabases(shard.first, shard.second, availability, cache)
            };
        });

    return result;
}

Shard::Databases makeShardDatabases(Shard::Id shardId, const RoleCache::Hosts& hosts,
        Availability::GetMethod availability, const Cache& cache) {
    using boost::adaptors::filtered;
    using boost::adaptors::transformed;
    struct Db {
        const Shard::Database::Address& address;
        const Shard::Database::Role role;
        const boost::optional<Shard::Database::Status> status;
        const StateCache::OptState state;
    };
    std::vector<Db> dbs;
    boost::transform(hosts, std::back_inserter(dbs),
        [&] (const RoleCache::Hosts::value_type& x) {
            const auto& address = x.first;
            const auto role = x.second;
            const auto status = cache.status.get(shardId, address, availability);
            const auto state = cache.state.get(shardId, address);
            return Db {address, role, status, state};
        });
    Shard::Databases result;
    boost::copy(dbs
        | filtered([&] (const Db& x) { return x.status.is_initialized() && x.state.is_initialized(); })
        | transformed([] (const Db& x) { return Shard::Database(x.address, x.role, x.status.get(), x.state.get()); }),
        std::inserter(result, result.end()));
    return result;
}

} // namespace cache
} // namespace sharpei
